<?php
require_once 'config/db.php';
require_once 'config/auth.php';
require_once 'config/functions.php';
require_once 'config/security.php';

// Initialize secure session
initSecureSession();

if (isLoggedIn()) {
    header("Location: index.php");
    exit();
}

$error = '';
$success = '';

if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    // Verify CSRF
    if (!verifyCSRFToken($_POST['csrf_token'] ?? '')) {
        $error = 'Invalid request. Please refresh and try again.';
    } else {
        $mobile = trim($_POST['mobile'] ?? '');
        $password = $_POST['password'] ?? '';
        $captcha = $_POST['captcha'] ?? '';
        
        if (empty($mobile) || empty($password) || empty($captcha)) {
            $error = 'All fields are required';
        } elseif (!verifyUserCaptcha($captcha)) {
            $error = 'Wrong captcha answer';
        } else {
            // Check IP-based rate limit
            $rateCheck = checkLoginRateLimit($pdo);
            if (!$rateCheck['allowed']) {
                $error = $rateCheck['message'];
            } else {
                // Check if account is locked
                $lockCheck = isAccountLocked($pdo, $mobile);
                if ($lockCheck['locked']) {
                    $error = "Account locked! Try again in {$lockCheck['remaining_minutes']} minutes.";
                    logSecurityEvent(null, 'LOGIN_ATTEMPT_LOCKED', 'medium', "Login attempt on locked account: $mobile");
                } else {
                    $stmt = $pdo->prepare("SELECT * FROM users WHERE mobile = ?");
                    $stmt->execute([$mobile]);
                    $user = $stmt->fetch(PDO::FETCH_ASSOC);
                    
                    if ($user && password_verify($password, $user['password'])) {
                        if ($user['status'] == 'pending') {
                            $error = 'Your account is pending approval';
                            recordLoginAttempt($pdo, $mobile, false);
                        } elseif ($user['status'] == 'blocked' || $user['status'] == 'rejected') {
                            $error = 'Your account has been blocked';
                            recordLoginAttempt($pdo, $mobile, false);
                        } else {
                            // Successful login
                            clearUserCaptcha();
                            recordLoginAttempt($pdo, $mobile, true);
                            
                            // Regenerate session ID for security
                            session_regenerate_id(true);
                            
                            $_SESSION['user_id'] = $user['id'];
                            $_SESSION['is_admin'] = $user['is_admin'] ?? 0;
                            $_SESSION['last_activity'] = time();
                            $_SESSION['session_created'] = time();
                            
                            // Track session
                            trackUserSession($pdo, $user['id']);
                            
                            // Log successful login
                            logSecurityEvent($user['id'], 'LOGIN_SUCCESS', 'low', 'User logged in successfully');
                            
                            if ($_SESSION['is_admin'] == 1) {
                                header("Location: admin/index.php");
                            } else {
                                header("Location: index.php");
                            }
                            exit();
                        }
                    } else {
                        $error = 'Invalid mobile or password';
                        recordLoginAttempt($pdo, $mobile, false);
                        
                        // Check remaining attempts
                        if ($user) {
                            $maxAttempts = getSetting($pdo, 'max_login_attempts', 5);
                            $remainingAttempts = max(0, $maxAttempts - ($user['failed_login_attempts'] + 1));
                            if ($remainingAttempts > 0 && $remainingAttempts <= 3) {
                                $error .= " ($remainingAttempts attempts remaining)";
                            }
                        }
                    }
                }
            }
        }
    }
}

// Generate captcha for display
$captchaText = generateUserCaptcha();
$csrfToken = generateCSRFToken();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Login - Satta King</title>
    <link rel="stylesheet" href="assets/css/style.css">
</head>
<body>
    <div class="auth-container">
        <div class="auth-box">
            <h1>🎰 SATTA KING</h1>
            
            <?php if ($error): ?>
                <div class="alert alert-error"><?= htmlspecialchars($error) ?></div>
            <?php endif; ?>
            
            <?php if ($success): ?>
                <div class="alert alert-success"><?= htmlspecialchars($success) ?></div>
            <?php endif; ?>
            
            <form method="POST">
                <input type="hidden" name="csrf_token" value="<?= $csrfToken ?>">
                
                <div class="form-group">
                    <label>Mobile Number</label>
                    <input type="tel" name="mobile" placeholder="Enter mobile number" maxlength="10" value="<?= htmlspecialchars($_POST['mobile'] ?? '') ?>" required>
                </div>
                
                <div class="form-group">
                    <label>Password</label>
                    <input type="password" name="password" placeholder="Enter password" required>
                </div>
                
                <div class="form-group">
                    <label>🔢 Captcha: <strong style="color:#ffd700;"><?= $captchaText ?></strong></label>
                    <input type="number" name="captcha" placeholder="Enter answer" required>
                </div>
                
                <button type="submit" class="btn">LOGIN</button>
            </form>
            
            <p class="auth-link">Don't have account? <a href="register.php">Register</a></p>
            
            <p class="auth-link" style="font-size:12px;color:#888;margin-top:15px;">
                Forgot password? Contact admin via WhatsApp
            </p>
        </div>
    </div>
</body>
</html>
