<?php
/**
 * Wallet API - Balance, Deposit, Withdraw, Transaction History
 * Endpoints:
 *   GET  /api/wallet.php?action=balance
 *   POST /api/wallet.php?action=deposit
 *   POST /api/wallet.php?action=withdraw
 *   GET  /api/wallet.php?action=history
 *   GET  /api/wallet.php?action=requests
 *   POST /api/wallet.php?action=set_pin
 *   POST /api/wallet.php?action=verify_pin
 */

require_once 'config.php';

$action = $_GET['action'] ?? '';

switch ($action) {
    case 'balance':
        handleGetBalance($pdo);
        break;
    case 'deposit':
        handleDepositRequest($pdo);
        break;
    case 'withdraw':
        handleWithdrawRequest($pdo);
        break;
    case 'history':
        handleTransactionHistory($pdo);
        break;
    case 'requests':
        handleWalletRequests($pdo);
        break;
    case 'set_pin':
        handleSetPin($pdo);
        break;
    case 'verify_pin':
        handleVerifyPin($pdo);
        break;
    case 'limits':
        handleGetLimits($pdo);
        break;
    default:
        sendError('Invalid action', 400);
}

/**
 * Get Wallet Balance
 */
function handleGetBalance($pdo) {
    $user = authenticate($pdo);
    
    $balance = getWalletBalance($pdo, $user['id']);
    
    // Get today's stats
    $stmt = $pdo->prepare("SELECT 
        COALESCE(SUM(CASE WHEN type = 'credit' THEN amount ELSE 0 END), 0) as today_credits,
        COALESCE(SUM(CASE WHEN type = 'debit' THEN amount ELSE 0 END), 0) as today_debits
        FROM wallet WHERE user_id = ? AND DATE(created_at) = CURDATE()");
    $stmt->execute([$user['id']]);
    $todayStats = $stmt->fetch(PDO::FETCH_ASSOC);
    
    // Get pending requests
    $stmt = $pdo->prepare("SELECT 
        COUNT(CASE WHEN type = 'deposit' THEN 1 END) as pending_deposits,
        COUNT(CASE WHEN type = 'withdraw' THEN 1 END) as pending_withdrawals,
        COALESCE(SUM(CASE WHEN type = 'deposit' THEN amount ELSE 0 END), 0) as pending_deposit_amount,
        COALESCE(SUM(CASE WHEN type = 'withdraw' THEN amount ELSE 0 END), 0) as pending_withdraw_amount
        FROM wallet_requests WHERE user_id = ? AND status = 'pending'");
    $stmt->execute([$user['id']]);
    $pending = $stmt->fetch(PDO::FETCH_ASSOC);
    
    // Daily withdraw limit status
    $dailyLimit = (int)getSetting($pdo, 'daily_withdraw_limit', '50000');
    $stmt = $pdo->prepare("SELECT COALESCE(daily_withdraw_total, 0) as today_withdrawn, daily_withdraw_date FROM users WHERE id = ?");
    $stmt->execute([$user['id']]);
    $withdrawStatus = $stmt->fetch(PDO::FETCH_ASSOC);
    
    $todayWithdrawn = 0;
    if ($withdrawStatus['daily_withdraw_date'] === date('Y-m-d')) {
        $todayWithdrawn = $withdrawStatus['today_withdrawn'];
    }
    
    sendResponse([
        'balance' => (float)$balance,
        'today' => [
            'credits' => (float)$todayStats['today_credits'],
            'debits' => (float)$todayStats['today_debits']
        ],
        'pending' => [
            'deposits' => (int)$pending['pending_deposits'],
            'withdrawals' => (int)$pending['pending_withdrawals'],
            'deposit_amount' => (float)$pending['pending_deposit_amount'],
            'withdraw_amount' => (float)$pending['pending_withdraw_amount']
        ],
        'limits' => [
            'daily_withdraw_limit' => $dailyLimit,
            'today_withdrawn' => (float)$todayWithdrawn,
            'remaining_withdraw' => max(0, $dailyLimit - $todayWithdrawn)
        ],
        'has_pin' => !empty($user['transaction_pin'])
    ]);
}

/**
 * Submit Deposit Request
 */
function handleDepositRequest($pdo) {
    $user = authenticate($pdo);
    
    // Check for file upload
    if (!isset($_FILES['screenshot']) && !isset($_POST['screenshot_base64'])) {
        $data = getJsonInput();
        if (!isset($data['amount'])) {
            sendError('Amount and payment screenshot required', 400);
        }
    }
    
    $amount = floatval($_POST['amount'] ?? $_FILES['amount'] ?? getJsonInput()['amount'] ?? 0);
    
    // Validate amount
    $minDeposit = (int)getSetting($pdo, 'min_deposit', '100');
    $maxDeposit = (int)getSetting($pdo, 'max_deposit', '50000');
    
    if ($amount < $minDeposit) {
        sendError("Minimum deposit is ₹{$minDeposit}", 400);
    }
    if ($amount > $maxDeposit) {
        sendError("Maximum deposit is ₹{$maxDeposit}", 400);
    }
    
    // Handle screenshot upload
    $screenshot = null;
    
    if (isset($_FILES['screenshot']) && $_FILES['screenshot']['error'] === 0) {
        $file = $_FILES['screenshot'];
        $ext = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
        $allowed = ['jpg', 'jpeg', 'png', 'gif'];
        
        if (!in_array($ext, $allowed)) {
            sendError('Invalid file type. Use JPG, PNG or GIF.', 400);
        }
        
        if ($file['size'] > 5 * 1024 * 1024) {
            sendError('File too large. Max 5MB allowed.', 400);
        }
        
        $filename = 'deposit_' . $user['id'] . '_' . time() . '.' . $ext;
        $uploadPath = __DIR__ . '/../uploads/' . $filename;
        
        if (move_uploaded_file($file['tmp_name'], $uploadPath)) {
            $screenshot = $filename;
        } else {
            sendError('Failed to upload screenshot', 500);
        }
    }
    
    // Rate limiting
    if (!checkApiRateLimit($pdo, 'deposit_' . $user['id'], 5, 3600)) {
        sendError('Too many deposit requests. Try again later.', 429);
    }
    
    // Create deposit request
    $stmt = $pdo->prepare("INSERT INTO wallet_requests (user_id, user_name_archive, user_mobile_archive, type, amount, screenshot, status, created_at) VALUES (?, ?, ?, 'deposit', ?, ?, 'pending', NOW())");
    $stmt->execute([
        $user['id'],
        $user['name'],
        $user['mobile'],
        $amount,
        $screenshot
    ]);
    $requestId = $pdo->lastInsertId();
    
    // Add notification
    $stmt = $pdo->prepare("INSERT INTO notifications (user_id, type, title, message, created_at) VALUES (?, 'deposit_request', 'Deposit Request Submitted 💰', ?, NOW())");
    $stmt->execute([
        $user['id'],
        "Your deposit request of ₹" . number_format($amount, 0) . " has been submitted and is pending approval."
    ]);
    
    sendResponse([
        'request_id' => (int)$requestId,
        'amount' => (float)$amount,
        'status' => 'pending',
        'message' => 'Deposit request submitted successfully. Waiting for admin approval.'
    ], 201);
}

/**
 * Submit Withdraw Request
 */
function handleWithdrawRequest($pdo) {
    $user = authenticate($pdo);
    $data = getJsonInput();
    
    validateRequired($data, ['amount']);
    
    // Check if PIN is required
    $requirePin = getSetting($pdo, 'require_transaction_pin', '1');
    if ($requirePin == '1') {
        if (empty($user['transaction_pin'])) {
            sendError('Please set a transaction PIN first', 400, ['code' => 'PIN_NOT_SET']);
        }
        
        if (!isset($data['pin'])) {
            sendError('Transaction PIN required', 400, ['code' => 'PIN_REQUIRED']);
        }
        
        if (!password_verify($data['pin'], $user['transaction_pin'])) {
            sendError('Invalid transaction PIN', 401, ['code' => 'INVALID_PIN']);
        }
    }
    
    $amount = floatval($data['amount']);
    
    // Validate amount
    $minWithdraw = (int)getSetting($pdo, 'min_withdraw', '100');
    $maxWithdraw = (int)getSetting($pdo, 'max_withdraw', '50000');
    
    if ($amount < $minWithdraw) {
        sendError("Minimum withdrawal is ₹{$minWithdraw}", 400);
    }
    if ($amount > $maxWithdraw) {
        sendError("Maximum withdrawal is ₹{$maxWithdraw}", 400);
    }
    
    // Check balance
    $balance = getWalletBalance($pdo, $user['id']);
    if ($amount > $balance) {
        sendError('Insufficient balance', 400);
    }
    
    // Check daily limit
    $dailyLimit = (int)getSetting($pdo, 'daily_withdraw_limit', '50000');
    $todayWithdrawn = 0;
    
    if ($user['daily_withdraw_date'] === date('Y-m-d')) {
        $todayWithdrawn = floatval($user['daily_withdraw_total'] ?? 0);
    }
    
    if ($todayWithdrawn + $amount > $dailyLimit) {
        $remaining = $dailyLimit - $todayWithdrawn;
        sendError("Daily withdrawal limit exceeded. Remaining: ₹" . number_format($remaining, 0), 400);
    }
    
    // Check if bank details are set
    if (empty($user['bank_name']) || empty($user['account_no'])) {
        sendError('Please add bank details in your profile first', 400, ['code' => 'BANK_DETAILS_REQUIRED']);
    }
    
    // Rate limiting
    if (!checkApiRateLimit($pdo, 'withdraw_' . $user['id'], 3, 3600)) {
        sendError('Too many withdrawal requests. Try again later.', 429);
    }
    
    // Create withdraw request
    $pdo->beginTransaction();
    try {
        // Insert request
        $stmt = $pdo->prepare("INSERT INTO wallet_requests (user_id, user_name_archive, user_mobile_archive, type, amount, status, created_at) VALUES (?, ?, ?, 'withdraw', ?, 'pending', NOW())");
        $stmt->execute([
            $user['id'],
            $user['name'],
            $user['mobile'],
            $amount
        ]);
        $requestId = $pdo->lastInsertId();
        
        // Hold amount in wallet (debit)
        $stmt = $pdo->prepare("INSERT INTO wallet (user_id, user_name_archive, user_mobile_archive, type, amount, description, reference_type, reference_id, created_at) VALUES (?, ?, ?, 'debit', ?, 'Withdrawal Request (On Hold)', 'withdraw_hold', ?, NOW())");
        $stmt->execute([
            $user['id'],
            $user['name'],
            $user['mobile'],
            $amount,
            $requestId
        ]);
        
        // Update daily withdraw total
        if ($user['daily_withdraw_date'] !== date('Y-m-d')) {
            $stmt = $pdo->prepare("UPDATE users SET daily_withdraw_total = ?, daily_withdraw_date = CURDATE() WHERE id = ?");
            $stmt->execute([$amount, $user['id']]);
        } else {
            $stmt = $pdo->prepare("UPDATE users SET daily_withdraw_total = daily_withdraw_total + ? WHERE id = ?");
            $stmt->execute([$amount, $user['id']]);
        }
        
        // Add notification
        $stmt = $pdo->prepare("INSERT INTO notifications (user_id, type, title, message, created_at) VALUES (?, 'withdraw_request', 'Withdrawal Request Submitted 💸', ?, NOW())");
        $stmt->execute([
            $user['id'],
            "Your withdrawal request of ₹" . number_format($amount, 0) . " has been submitted."
        ]);
        
        $pdo->commit();
        
        $newBalance = getWalletBalance($pdo, $user['id']);
        
        sendResponse([
            'request_id' => (int)$requestId,
            'amount' => (float)$amount,
            'status' => 'pending',
            'new_balance' => (float)$newBalance,
            'message' => 'Withdrawal request submitted successfully.'
        ], 201);
        
    } catch (Exception $e) {
        $pdo->rollBack();
        sendError('Failed to submit withdrawal request', 500);
    }
}

/**
 * Get Transaction History
 */
function handleTransactionHistory($pdo) {
    $user = authenticate($pdo);
    
    $type = $_GET['type'] ?? ''; // credit, debit
    $limit = min(intval($_GET['limit'] ?? 50), 100);
    $page = max(intval($_GET['page'] ?? 1), 1);
    $offset = ($page - 1) * $limit;
    
    // Build query
    $where = ["user_id = ?"];
    $params = [$user['id']];
    
    if ($type && in_array($type, ['credit', 'debit'])) {
        $where[] = "type = ?";
        $params[] = $type;
    }
    
    $whereClause = implode(' AND ', $where);
    
    // Get total
    $stmt = $pdo->prepare("SELECT COUNT(*) FROM wallet WHERE {$whereClause}");
    $stmt->execute($params);
    $total = $stmt->fetchColumn();
    
    // Get transactions
    $params[] = $limit;
    $params[] = $offset;
    $stmt = $pdo->prepare("SELECT * FROM wallet WHERE {$whereClause} ORDER BY created_at DESC LIMIT ? OFFSET ?");
    $stmt->execute($params);
    $transactions = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    $txList = [];
    foreach ($transactions as $tx) {
        $txList[] = [
            'id' => (int)$tx['id'],
            'type' => $tx['type'],
            'amount' => (float)$tx['amount'],
            'description' => $tx['description'],
            'reference_type' => $tx['reference_type'],
            'reference_id' => $tx['reference_id'],
            'created_at' => $tx['created_at']
        ];
    }
    
    sendResponse([
        'transactions' => $txList,
        'pagination' => [
            'current_page' => $page,
            'per_page' => $limit,
            'total' => (int)$total,
            'total_pages' => ceil($total / $limit)
        ]
    ]);
}

/**
 * Get Wallet Requests (Deposits/Withdrawals)
 */
function handleWalletRequests($pdo) {
    $user = authenticate($pdo);
    
    $type = $_GET['type'] ?? ''; // deposit, withdraw
    $status = $_GET['status'] ?? ''; // pending, approved, rejected
    $limit = min(intval($_GET['limit'] ?? 20), 50);
    $page = max(intval($_GET['page'] ?? 1), 1);
    $offset = ($page - 1) * $limit;
    
    // Build query
    $where = ["user_id = ?"];
    $params = [$user['id']];
    
    if ($type && in_array($type, ['deposit', 'withdraw'])) {
        $where[] = "type = ?";
        $params[] = $type;
    }
    
    if ($status && in_array($status, ['pending', 'approved', 'rejected'])) {
        $where[] = "status = ?";
        $params[] = $status;
    }
    
    $whereClause = implode(' AND ', $where);
    
    // Get total
    $stmt = $pdo->prepare("SELECT COUNT(*) FROM wallet_requests WHERE {$whereClause}");
    $stmt->execute($params);
    $total = $stmt->fetchColumn();
    
    // Get requests
    $params[] = $limit;
    $params[] = $offset;
    $stmt = $pdo->prepare("SELECT * FROM wallet_requests WHERE {$whereClause} ORDER BY created_at DESC LIMIT ? OFFSET ?");
    $stmt->execute($params);
    $requests = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    $requestList = [];
    foreach ($requests as $req) {
        $requestList[] = [
            'id' => (int)$req['id'],
            'type' => $req['type'],
            'amount' => (float)$req['amount'],
            'status' => $req['status'],
            'admin_note' => $req['admin_note'],
            'screenshot' => $req['screenshot'] ? '/uploads/' . $req['screenshot'] : null,
            'created_at' => $req['created_at'],
            'processed_at' => $req['processed_at']
        ];
    }
    
    sendResponse([
        'requests' => $requestList,
        'pagination' => [
            'current_page' => $page,
            'per_page' => $limit,
            'total' => (int)$total,
            'total_pages' => ceil($total / $limit)
        ]
    ]);
}

/**
 * Set Transaction PIN
 */
function handleSetPin($pdo) {
    $user = authenticate($pdo);
    $data = getJsonInput();
    
    validateRequired($data, ['pin']);
    
    $pin = $data['pin'];
    
    // Validate PIN format (4 digits)
    if (!preg_match('/^[0-9]{4}$/', $pin)) {
        sendError('PIN must be exactly 4 digits', 400);
    }
    
    // If already has PIN, require current PIN
    if (!empty($user['transaction_pin'])) {
        if (!isset($data['current_pin'])) {
            sendError('Current PIN required to change PIN', 400);
        }
        if (!password_verify($data['current_pin'], $user['transaction_pin'])) {
            sendError('Current PIN is incorrect', 401);
        }
    }
    
    // Hash and save PIN
    $hashedPin = password_hash($pin, PASSWORD_DEFAULT);
    $stmt = $pdo->prepare("UPDATE users SET transaction_pin = ? WHERE id = ?");
    $stmt->execute([$hashedPin, $user['id']]);
    
    sendResponse([
        'message' => 'Transaction PIN set successfully'
    ]);
}

/**
 * Verify Transaction PIN
 */
function handleVerifyPin($pdo) {
    $user = authenticate($pdo);
    $data = getJsonInput();
    
    validateRequired($data, ['pin']);
    
    if (empty($user['transaction_pin'])) {
        sendError('Transaction PIN not set', 400, ['code' => 'PIN_NOT_SET']);
    }
    
    if (!password_verify($data['pin'], $user['transaction_pin'])) {
        sendError('Invalid PIN', 401);
    }
    
    sendResponse([
        'valid' => true,
        'message' => 'PIN verified successfully'
    ]);
}

/**
 * Get Wallet Limits
 */
function handleGetLimits($pdo) {
    $user = authenticate($pdo);
    
    sendResponse([
        'deposit' => [
            'min' => (int)getSetting($pdo, 'min_deposit', '100'),
            'max' => (int)getSetting($pdo, 'max_deposit', '50000')
        ],
        'withdraw' => [
            'min' => (int)getSetting($pdo, 'min_withdraw', '100'),
            'max' => (int)getSetting($pdo, 'max_withdraw', '50000'),
            'daily_limit' => (int)getSetting($pdo, 'daily_withdraw_limit', '50000')
        ],
        'require_pin' => getSetting($pdo, 'require_transaction_pin', '1') == '1'
    ]);
}
?>
