<?php
/**
 * Notifications API - Get notifications, Mark as read
 * Endpoints:
 *   GET  /api/notifications.php?action=list
 *   POST /api/notifications.php?action=mark_read
 *   GET  /api/notifications.php?action=unread_count
 */

require_once 'config.php';

$action = $_GET['action'] ?? 'list';

switch ($action) {
    case 'list':
        handleGetNotifications($pdo);
        break;
    case 'mark_read':
        handleMarkRead($pdo);
        break;
    case 'unread_count':
        handleUnreadCount($pdo);
        break;
    case 'delete':
        handleDeleteNotification($pdo);
        break;
    case 'clear_all':
        handleClearAll($pdo);
        break;
    default:
        sendError('Invalid action', 400);
}

/**
 * Get Notifications List
 */
function handleGetNotifications($pdo) {
    $user = authenticate($pdo);
    
    $unreadOnly = isset($_GET['unread']) && $_GET['unread'] == '1';
    $limit = min(intval($_GET['limit'] ?? 50), 100);
    $page = max(intval($_GET['page'] ?? 1), 1);
    $offset = ($page - 1) * $limit;
    
    // Build query
    $where = ["user_id = ?"];
    $params = [$user['id']];
    
    if ($unreadOnly) {
        $where[] = "is_read = 0";
    }
    
    $whereClause = implode(' AND ', $where);
    
    // Get total
    $stmt = $pdo->prepare("SELECT COUNT(*) FROM notifications WHERE {$whereClause}");
    $stmt->execute($params);
    $total = $stmt->fetchColumn();
    
    // Get notifications
    $params[] = $limit;
    $params[] = $offset;
    $stmt = $pdo->prepare("SELECT * FROM notifications WHERE {$whereClause} ORDER BY created_at DESC LIMIT ? OFFSET ?");
    $stmt->execute($params);
    $notifications = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    $notificationsList = [];
    foreach ($notifications as $notif) {
        $notificationsList[] = [
            'id' => (int)$notif['id'],
            'type' => $notif['type'],
            'title' => $notif['title'],
            'message' => $notif['message'],
            'data' => $notif['data'] ? json_decode($notif['data'], true) : null,
            'is_read' => (bool)$notif['is_read'],
            'created_at' => $notif['created_at'],
            'time_ago' => getTimeAgo($notif['created_at'])
        ];
    }
    
    // Get unread count
    $stmt = $pdo->prepare("SELECT COUNT(*) FROM notifications WHERE user_id = ? AND is_read = 0");
    $stmt->execute([$user['id']]);
    $unreadCount = $stmt->fetchColumn();
    
    sendResponse([
        'notifications' => $notificationsList,
        'unread_count' => (int)$unreadCount,
        'pagination' => [
            'current_page' => $page,
            'per_page' => $limit,
            'total' => (int)$total,
            'total_pages' => ceil($total / $limit)
        ]
    ]);
}

/**
 * Mark Notification(s) as Read
 */
function handleMarkRead($pdo) {
    $user = authenticate($pdo);
    $data = getJsonInput();
    
    if (isset($data['notification_id'])) {
        // Mark single notification
        $notifId = intval($data['notification_id']);
        $stmt = $pdo->prepare("UPDATE notifications SET is_read = 1 WHERE id = ? AND user_id = ?");
        $stmt->execute([$notifId, $user['id']]);
        
        sendResponse(['message' => 'Notification marked as read']);
    } elseif (isset($data['all']) && $data['all'] === true) {
        // Mark all as read
        $stmt = $pdo->prepare("UPDATE notifications SET is_read = 1 WHERE user_id = ? AND is_read = 0");
        $stmt->execute([$user['id']]);
        
        sendResponse(['message' => 'All notifications marked as read']);
    } else {
        sendError('Provide notification_id or set all=true', 400);
    }
}

/**
 * Get Unread Count Only
 */
function handleUnreadCount($pdo) {
    $user = authenticate($pdo);
    
    $stmt = $pdo->prepare("SELECT COUNT(*) FROM notifications WHERE user_id = ? AND is_read = 0");
    $stmt->execute([$user['id']]);
    $count = $stmt->fetchColumn();
    
    sendResponse([
        'unread_count' => (int)$count
    ]);
}

/**
 * Delete Single Notification
 */
function handleDeleteNotification($pdo) {
    $user = authenticate($pdo);
    $data = getJsonInput();
    
    validateRequired($data, ['notification_id']);
    
    $notifId = intval($data['notification_id']);
    
    $stmt = $pdo->prepare("DELETE FROM notifications WHERE id = ? AND user_id = ?");
    $stmt->execute([$notifId, $user['id']]);
    
    if ($stmt->rowCount() === 0) {
        sendError('Notification not found', 404);
    }
    
    sendResponse(['message' => 'Notification deleted']);
}

/**
 * Clear All Notifications
 */
function handleClearAll($pdo) {
    $user = authenticate($pdo);
    
    $stmt = $pdo->prepare("DELETE FROM notifications WHERE user_id = ?");
    $stmt->execute([$user['id']]);
    
    sendResponse([
        'message' => 'All notifications cleared',
        'deleted' => $stmt->rowCount()
    ]);
}

/**
 * Helper: Get time ago string
 */
function getTimeAgo($datetime) {
    $now = new DateTime();
    $ago = new DateTime($datetime);
    $diff = $now->diff($ago);
    
    if ($diff->y > 0) return $diff->y . ' year' . ($diff->y > 1 ? 's' : '') . ' ago';
    if ($diff->m > 0) return $diff->m . ' month' . ($diff->m > 1 ? 's' : '') . ' ago';
    if ($diff->d > 0) return $diff->d . ' day' . ($diff->d > 1 ? 's' : '') . ' ago';
    if ($diff->h > 0) return $diff->h . ' hour' . ($diff->h > 1 ? 's' : '') . ' ago';
    if ($diff->i > 0) return $diff->i . ' minute' . ($diff->i > 1 ? 's' : '') . ' ago';
    return 'Just now';
}
?>
