<?php
/**
 * API Configuration & Helper Functions
 * Satta King Mobile App API
 */

// Error reporting for development (disable in production)
error_reporting(0);
ini_set('display_errors', 0);

// CORS Headers for mobile app
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization, X-Requested-With');
header('Content-Type: application/json; charset=UTF-8');

// Handle preflight requests
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

// Include database connection
require_once __DIR__ . '/../config/db.php';
require_once __DIR__ . '/../config/functions.php';

// JWT Configuration
define('JWT_SECRET', 'SattaKing_SecretKey_2024_XyZ_!@#$%'); // Change this in production!
define('JWT_EXPIRY', 86400 * 30); // 30 days

/**
 * Generate JWT Token
 */
function generateJWT($userId, $mobile) {
    $header = json_encode(['typ' => 'JWT', 'alg' => 'HS256']);
    $payload = json_encode([
        'user_id' => $userId,
        'mobile' => $mobile,
        'iat' => time(),
        'exp' => time() + JWT_EXPIRY
    ]);
    
    $base64Header = base64UrlEncode($header);
    $base64Payload = base64UrlEncode($payload);
    
    $signature = hash_hmac('sha256', $base64Header . '.' . $base64Payload, JWT_SECRET, true);
    $base64Signature = base64UrlEncode($signature);
    
    return $base64Header . '.' . $base64Payload . '.' . $base64Signature;
}

/**
 * Verify JWT Token
 */
function verifyJWT($token) {
    $parts = explode('.', $token);
    if (count($parts) !== 3) {
        return false;
    }
    
    list($base64Header, $base64Payload, $base64Signature) = $parts;
    
    $signature = hash_hmac('sha256', $base64Header . '.' . $base64Payload, JWT_SECRET, true);
    $expectedSignature = base64UrlEncode($signature);
    
    if (!hash_equals($expectedSignature, $base64Signature)) {
        return false;
    }
    
    $payload = json_decode(base64UrlDecode($base64Payload), true);
    
    if (!$payload || !isset($payload['exp']) || $payload['exp'] < time()) {
        return false;
    }
    
    return $payload;
}

/**
 * Base64 URL Encode
 */
function base64UrlEncode($data) {
    return rtrim(strtr(base64_encode($data), '+/', '-_'), '=');
}

/**
 * Base64 URL Decode
 */
function base64UrlDecode($data) {
    return base64_decode(strtr($data, '-_', '+/'));
}

/**
 * Get Authorization Token from Header
 */
function getBearerToken() {
    $headers = getallheaders();
    
    // Check various header formats
    $authHeader = $headers['Authorization'] ?? $headers['authorization'] ?? '';
    
    if (empty($authHeader)) {
        // Try Apache-specific
        if (isset($_SERVER['HTTP_AUTHORIZATION'])) {
            $authHeader = $_SERVER['HTTP_AUTHORIZATION'];
        } elseif (isset($_SERVER['REDIRECT_HTTP_AUTHORIZATION'])) {
            $authHeader = $_SERVER['REDIRECT_HTTP_AUTHORIZATION'];
        }
    }
    
    if (preg_match('/Bearer\s+(.*)$/i', $authHeader, $matches)) {
        return $matches[1];
    }
    
    return null;
}

/**
 * Authenticate Request - Returns user data or sends error
 */
function authenticate($pdo) {
    $token = getBearerToken();
    
    if (!$token) {
        sendError('Authorization token required', 401);
    }
    
    $payload = verifyJWT($token);
    
    if (!$payload) {
        sendError('Invalid or expired token', 401);
    }
    
    // Get user from database
    $stmt = $pdo->prepare("SELECT * FROM users WHERE id = ? AND status = 'active' AND is_admin = 0");
    $stmt->execute([$payload['user_id']]);
    $user = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$user) {
        sendError('User not found or inactive', 401);
    }
    
    return $user;
}

/**
 * Send JSON Response
 */
function sendResponse($data, $statusCode = 200) {
    http_response_code($statusCode);
    echo json_encode([
        'success' => true,
        'data' => $data
    ], JSON_UNESCAPED_UNICODE);
    exit();
}

/**
 * Send Error Response
 */
function sendError($message, $statusCode = 400, $errors = null) {
    http_response_code($statusCode);
    $response = [
        'success' => false,
        'message' => $message
    ];
    if ($errors) {
        $response['errors'] = $errors;
    }
    echo json_encode($response, JSON_UNESCAPED_UNICODE);
    exit();
}

/**
 * Get POST JSON Data
 */
function getJsonInput() {
    $json = file_get_contents('php://input');
    $data = json_decode($json, true);
    return $data ?? [];
}

/**
 * Validate Required Fields
 */
function validateRequired($data, $fields) {
    $missing = [];
    foreach ($fields as $field) {
        if (!isset($data[$field]) || trim($data[$field]) === '') {
            $missing[] = $field;
        }
    }
    if (!empty($missing)) {
        sendError('Missing required fields: ' . implode(', ', $missing), 400);
    }
}

/**
 * Mask Mobile Number
 */
function maskMobile($mobile) {
    if (strlen($mobile) >= 10) {
        return substr($mobile, 0, 2) . 'XXXXXX' . substr($mobile, -2);
    }
    return $mobile;
}

/**
 * Format Currency
 */
function formatCurrency($amount) {
    return '₹' . number_format($amount, 2);
}

/**
 * Log API Request (for debugging)
 */
function logApiRequest($endpoint, $userId = null) {
    global $pdo;
    // Optionally log API requests
    // You can implement this if needed
}

/**
 * Rate Limit Check for API
 */
function checkApiRateLimit($pdo, $identifier, $limit = 60, $window = 60) {
    $stmt = $pdo->prepare("SELECT count, window_start FROM rate_limits WHERE identifier = ? AND action_type = 'api'");
    $stmt->execute([$identifier]);
    $record = $stmt->fetch(PDO::FETCH_ASSOC);
    
    $now = time();
    
    if ($record) {
        $windowStart = strtotime($record['window_start']);
        if ($now - $windowStart < $window) {
            if ($record['count'] >= $limit) {
                return false;
            }
            $stmt = $pdo->prepare("UPDATE rate_limits SET count = count + 1 WHERE identifier = ? AND action_type = 'api'");
            $stmt->execute([$identifier]);
        } else {
            $stmt = $pdo->prepare("UPDATE rate_limits SET count = 1, window_start = NOW() WHERE identifier = ? AND action_type = 'api'");
            $stmt->execute([$identifier]);
        }
    } else {
        $stmt = $pdo->prepare("INSERT INTO rate_limits (identifier, action_type, count, window_start) VALUES (?, 'api', 1, NOW())");
        $stmt->execute([$identifier]);
    }
    
    return true;
}

/**
 * Get Client IP
 */
function getClientIP() {
    if (!empty($_SERVER['HTTP_CLIENT_IP'])) {
        return $_SERVER['HTTP_CLIENT_IP'];
    } elseif (!empty($_SERVER['HTTP_X_FORWARDED_FOR'])) {
        return explode(',', $_SERVER['HTTP_X_FORWARDED_FOR'])[0];
    } else {
        return $_SERVER['REMOTE_ADDR'] ?? '0.0.0.0';
    }
}
?>
