<?php
require_once '../debug.php';
require_once '../config/db.php';
require_once '../config/auth.php';
require_once '../config/functions.php';
require_once '../config/security.php';

initSecureSession();
requireAdmin();

// Check session timeout
if (!checkSessionTimeout($pdo)) {
    header("Location: ../login.php?msg=Session+expired");
    exit();
}

$admin = getUser($pdo);

// Track admin session
trackUserSession($pdo, $admin['id']);

$error = '';
$success = '';

// Activity Log Function with archive names

// Calculation Log Function with archive names

// Handle Approve Deposit
if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['approve_deposit'])) {
    // Use inline captcha verification for quick approve buttons
    if (!verifyInlineCaptcha($_POST['captcha_answer'] ?? '')) {
        $error = getCaptchaError();
    } else {
        $request_id = $_POST['request_id'];
    
    $stmt = $pdo->prepare("SELECT wr.*, u.name as user_name, u.mobile as user_mobile FROM wallet_requests wr JOIN users u ON wr.user_id = u.id WHERE wr.id = ? AND wr.status = 'pending'");
    $stmt->execute([$request_id]);
    $request = $stmt->fetch();
    
    if ($request) {
        $pdo->beginTransaction();
        try {
            $balance_before = getWalletBalance($pdo, $request['user_id']);
            
            // Update request status with admin info
            $stmt = $pdo->prepare("UPDATE wallet_requests SET status = 'approved', admin_id = ?, admin_name_archive = ?, processed_at = NOW(), updated_at = NOW() WHERE id = ?");
            $stmt->execute([$admin['id'], $admin['name'], $request_id]);
            
            // Add wallet credit WITH archive columns
            $stmt = $pdo->prepare("INSERT INTO wallet (user_id, user_name_archive, user_mobile_archive, type, amount, description, reference_type, reference_id) VALUES (?, ?, ?, 'credit', ?, ?, 'wallet_requests', ?)");
            $stmt->execute([$request['user_id'], $request['user_name'], $request['user_mobile'], $request['amount'], 'Deposit Approved', $request_id]);
            $wallet_id = $pdo->lastInsertId();
            
            $balance_after = getWalletBalance($pdo, $request['user_id']);
            
            // Log Calculation
            logCalculation($pdo, $request['user_id'], 'DEPOSIT_APPROVED', 'wallet_requests', $request_id, 
                $request['amount'], $balance_before, $balance_after,
                "Balance ({$balance_before}) + Deposit ({$request['amount']}) = {$balance_after}",
                "Deposit approved for {$request['user_name']}", $admin['id']
            );
            
            // Log Activity
            logActivity($pdo, $admin['id'], 'APPROVE_DEPOSIT', 'wallet_requests', $request_id,
                ['status' => 'pending'],
                ['status' => 'approved', 'amount' => $request['amount']],
                "Deposit of ₹{$request['amount']} approved for {$request['user_name']}"
            );
            
            // Add notification for user
            addNotification($pdo, $request['user_id'], 'deposit_approved', 'Deposit Approved! 💰', 
                "Your deposit of ₹" . number_format($request['amount'], 0) . " has been approved and added to your wallet.");
            
            $pdo->commit();
            $success = 'Deposit of ₹' . number_format($request['amount'], 0) . ' approved!';
        } catch (Exception $e) {
            $pdo->rollBack();
            $error = 'Error: ' . $e->getMessage();
        }
    }
    }
}

// Handle Reject Deposit/Withdraw
if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['reject_request'])) {
    if (!verifyAdminCaptcha($_POST['captcha_answer'] ?? '')) {
        $error = getCaptchaError();
    } else {
        $request_id = $_POST['request_id'];
        $reason = $_POST['reason'] ?? 'Request rejected';
    
    $stmt = $pdo->prepare("SELECT wr.*, u.name as user_name, u.mobile as user_mobile FROM wallet_requests wr JOIN users u ON wr.user_id = u.id WHERE wr.id = ?");
    $stmt->execute([$request_id]);
    $request = $stmt->fetch();
    
    if ($request && $request['status'] == 'pending') {
        $pdo->beginTransaction();
        try {
            // Update request status
            $stmt = $pdo->prepare("UPDATE wallet_requests SET status = 'rejected', admin_id = ?, admin_name_archive = ?, admin_note = ?, processed_at = NOW(), updated_at = NOW() WHERE id = ?");
            $stmt->execute([$admin['id'], $admin['name'], $reason, $request_id]);
            
            // If WITHDRAW request, credit back the held amount
            if ($request['type'] == 'withdraw') {
                // Delete the hold entry
                $stmt = $pdo->prepare("DELETE FROM wallet WHERE user_id = ? AND reference_type = 'withdraw_hold' AND reference_id = ?");
                $stmt->execute([$request['user_id'], $request_id]);
                
                // Log Activity
                logActivity($pdo, $admin['id'], 'REJECT_WITHDRAW', 'wallet_requests', $request_id,
                    ['status' => 'pending'],
                    ['status' => 'rejected', 'reason' => $reason],
                    "Withdrawal of ₹{$request['amount']} rejected for {$request['user_name']}. Amount refunded. Reason: $reason"
                );
                
                // Add notification
                addNotification($pdo, $request['user_id'], 'withdraw_rejected', 'Withdrawal Rejected ❌', 
                    "Your withdrawal of ₹" . number_format($request['amount'], 0) . " was rejected. Amount refunded to wallet. Reason: $reason");
            } else {
                // Log Activity for deposit reject
                logActivity($pdo, $admin['id'], 'REJECT_DEPOSIT', 'wallet_requests', $request_id,
                    ['status' => 'pending'],
                    ['status' => 'rejected', 'reason' => $reason],
                    "Deposit of ₹{$request['amount']} rejected for {$request['user_name']}. Reason: $reason"
                );
                
                // Add notification
                addNotification($pdo, $request['user_id'], 'deposit_rejected', 'Deposit Rejected ❌', 
                    "Your deposit of ₹" . number_format($request['amount'], 0) . " was rejected. Reason: $reason");
            }
            
            $pdo->commit();
            $success = $request['type'] == 'withdraw' ? 'Withdraw request rejected! Amount refunded to user.' : 'Deposit request rejected!';
            
        } catch (Exception $e) {
            $pdo->rollBack();
            $error = 'Error: ' . $e->getMessage();
        }
    }
    }
}

// Handle Approve Withdraw
if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['approve_withdraw'])) {
    // Use inline captcha verification for quick approve buttons
    if (!verifyInlineCaptcha($_POST['captcha_answer'] ?? '')) {
        $error = getCaptchaError();
    } else {
        $request_id = $_POST['request_id'];
    
    $stmt = $pdo->prepare("SELECT wr.*, u.name as user_name, u.mobile as user_mobile FROM wallet_requests wr JOIN users u ON wr.user_id = u.id WHERE wr.id = ? AND wr.status = 'pending'");
    $stmt->execute([$request_id]);
    $request = $stmt->fetch();
    
    if ($request) {
        $pdo->beginTransaction();
        try {
            // Update request status with admin info
            $stmt = $pdo->prepare("UPDATE wallet_requests SET status = 'approved', admin_id = ?, admin_name_archive = ?, processed_at = NOW(), updated_at = NOW() WHERE id = ?");
            $stmt->execute([$admin['id'], $admin['name'], $request_id]);
            
            // Update the held wallet entry description (already deducted when user made request)
            $stmt = $pdo->prepare("UPDATE wallet SET description = 'Withdrawal Approved', reference_type = 'wallet_requests' WHERE user_id = ? AND reference_type = 'withdraw_hold' AND reference_id = ?");
            $stmt->execute([$request['user_id'], $request_id]);
            
            $balance_after = getWalletBalance($pdo, $request['user_id']);
            
            // Log Activity
            logActivity($pdo, $admin['id'], 'APPROVE_WITHDRAW', 'wallet_requests', $request_id,
                ['status' => 'pending'],
                ['status' => 'approved', 'amount' => $request['amount']],
                "Withdrawal of ₹{$request['amount']} approved for {$request['user_name']}"
            );
            
            // Add notification for user
            addNotification($pdo, $request['user_id'], 'withdraw_approved', 'Withdrawal Approved! 💸', 
                "Your withdrawal of ₹" . number_format($request['amount'], 0) . " has been approved and will be processed soon.");
            
            $pdo->commit();
            $success = 'Withdrawal of ₹' . number_format($request['amount'], 0) . ' approved!';
        } catch (Exception $e) {
            $pdo->rollBack();
            $error = 'Error: ' . $e->getMessage();
        }
    }
    }
}

// Handle Manual Credit/Debit
if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['manual_transaction'])) {
    if (!verifyAdminCaptcha($_POST['captcha_answer'] ?? '')) {
        $error = getCaptchaError();
    } else {
        $user_id = $_POST['user_id'];
        $type = $_POST['txn_type'];
        $amount = floatval($_POST['amount']);
        $description = trim($_POST['description']);
    
    // Get user info
    $stmt = $pdo->prepare("SELECT name, mobile FROM users WHERE id = ?");
    $stmt->execute([$user_id]);
    $user = $stmt->fetch();
    
    if ($amount <= 0) {
        $error = 'Amount must be greater than 0';
    } elseif (empty($description)) {
        $error = 'Description is required';
    } elseif (!$user) {
        $error = 'User not found';
    } else {
        $balance_before = getWalletBalance($pdo, $user_id);
        
        if ($type == 'debit' && $balance_before < $amount) {
            $error = "User has insufficient balance! Current: ₹" . number_format($balance_before, 0);
        } else {
            $pdo->beginTransaction();
            try {
                // Insert WITH archive columns
                $stmt = $pdo->prepare("INSERT INTO wallet (user_id, user_name_archive, user_mobile_archive, type, amount, description) VALUES (?, ?, ?, ?, ?, ?)");
                $stmt->execute([$user_id, $user['name'], $user['mobile'], $type, $amount, $description]);
                $wallet_id = $pdo->lastInsertId();
                
                $balance_after = getWalletBalance($pdo, $user_id);
                
                // Log Calculation
                $formula = $type == 'credit' 
                    ? "Balance ({$balance_before}) + Credit ({$amount}) = {$balance_after}"
                    : "Balance ({$balance_before}) - Debit ({$amount}) = {$balance_after}";
                
                logCalculation($pdo, $user_id, 'MANUAL_' . strtoupper($type), 'wallet', $wallet_id,
                    $type == 'credit' ? $amount : -$amount, $balance_before, $balance_after,
                    $formula, "Manual $type: $description", $admin['id']
                );
                
                // Log Activity
                logActivity($pdo, $admin['id'], 'MANUAL_' . strtoupper($type), 'wallet', $wallet_id,
                    null, ['type' => $type, 'amount' => $amount, 'description' => $description],
                    "Manual $type of ₹$amount for {$user['name']}. Reason: $description"
                );
                
                $pdo->commit();
                $success = ucfirst($type) . ' of ₹' . number_format($amount, 0) . ' successful!';
            } catch (Exception $e) {
                $pdo->rollBack();
                $error = 'Error: ' . $e->getMessage();
            }
        }
    }
    }
}

// Filters
$filter_type = $_GET['type'] ?? '';
$filter_status = $_GET['status'] ?? 'pending';

$where = "WHERE 1=1";
$params = [];

if ($filter_type) {
    $where .= " AND wr.type = ?";
    $params[] = $filter_type;
}
if ($filter_status) {
    $where .= " AND wr.status = ?";
    $params[] = $filter_status;
}

// Get requests with user info (using archive columns as fallback)
$sql = "SELECT wr.*, 
        COALESCE(u.name, wr.user_name_archive, CONCAT('User #', wr.user_id)) as user_name, 
        COALESCE(u.mobile, wr.user_mobile_archive) as mobile,
        u.upi_id as user_upi,
        u.qr_code as user_qr,
        u.bank_name as user_bank,
        u.account_no as user_account,
        u.account_holder as user_account_holder,
        u.ifsc_code as user_ifsc
        FROM wallet_requests wr 
        LEFT JOIN users u ON wr.user_id = u.id 
        $where 
        ORDER BY wr.created_at DESC 
        LIMIT 100";
$stmt = $pdo->prepare($sql);
$stmt->execute($params);
$requests = $stmt->fetchAll();

// Get users for manual transaction
$stmt = $pdo->query("SELECT id, name, mobile FROM users WHERE is_admin = 0 AND status = 'active' ORDER BY name ASC");
$users = $stmt->fetchAll();

// Stats
$stmt = $pdo->query("SELECT COUNT(*) as c FROM wallet_requests WHERE status = 'pending' AND type = 'deposit'");
$pendingDeposits = $stmt->fetch()['c'];

$stmt = $pdo->query("SELECT COUNT(*) as c FROM wallet_requests WHERE status = 'pending' AND type = 'withdraw'");
$pendingWithdraws = $stmt->fetch()['c'];

$stmt = $pdo->query("SELECT COALESCE(SUM(amount), 0) as total FROM wallet_requests WHERE status = 'approved' AND type = 'deposit' AND DATE(created_at) = CURDATE()");
$todayDeposits = $stmt->fetch()['total'];

$stmt = $pdo->query("SELECT COALESCE(SUM(amount), 0) as total FROM wallet_requests WHERE status = 'approved' AND type = 'withdraw' AND DATE(created_at) = CURDATE()");
$todayWithdraws = $stmt->fetch()['total'];

// For sidebar
$stmt = $pdo->query("SELECT COUNT(*) as c FROM users WHERE is_admin = 0 AND status = 'pending'");
$pendingUsers = $stmt->fetch()['c'];
$pendingRequests = $pendingDeposits + $pendingWithdraws;
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Wallet - Admin Panel</title>
    <style>
        * { margin: 0; padding: 0; box-sizing: border-box; }
        body { font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif; background: #0f0f0f; color: #fff; min-height: 100vh; }
        .admin-layout { display: flex; min-height: 100vh; }
        .sidebar { width: 260px; background: #1a1a1a; border-right: 1px solid #333; position: fixed; top: 0; left: 0; bottom: 0; overflow-y: auto; z-index: 100; }
        .sidebar-header { padding: 20px; border-bottom: 1px solid #333; text-align: center; }
        .sidebar-logo { font-size: 22px; font-weight: 800; color: #ffd700; }
        .sidebar-subtitle { font-size: 11px; color: #888; margin-top: 5px; }
        .sidebar-nav { padding: 15px 0; }
        .nav-item { display: flex; align-items: center; gap: 12px; padding: 14px 20px; color: #888; text-decoration: none; transition: all 0.3s; border-left: 3px solid transparent; }
        .nav-item:hover { background: rgba(255,255,255,0.05); color: #fff; }
        .nav-item.active { background: rgba(139,0,0,0.3); color: #ffd700; border-left-color: #ffd700; }
        .nav-icon { font-size: 20px; width: 24px; text-align: center; }
        .nav-text { font-size: 14px; font-weight: 500; }
        .nav-badge { margin-left: auto; background: #ff4444; color: #fff; padding: 2px 8px; border-radius: 10px; font-size: 11px; font-weight: 700; }
        .main-content { flex: 1; margin-left: 260px; padding: 20px; }
        .top-bar { display: flex; justify-content: space-between; align-items: center; margin-bottom: 25px; padding-bottom: 20px; border-bottom: 1px solid #333; flex-wrap: wrap; gap: 15px; }
        .page-title { font-size: 24px; font-weight: 700; }
        .page-title span { color: #ffd700; }
        .btn-manual { padding: 12px 25px; background: linear-gradient(135deg, #ffd700 0%, #ff8c00 100%); border: none; border-radius: 10px; color: #000; font-weight: 700; cursor: pointer; font-size: 14px; }
        .alert { padding: 15px; border-radius: 10px; margin-bottom: 20px; font-size: 14px; }
        .alert.success { background: rgba(0,255,0,0.2); border: 1px solid #00cc00; color: #00ff00; }
        .alert.error { background: rgba(255,0,0,0.2); border: 1px solid #ff4444; color: #ff6666; }
        .stats-row { display: grid; grid-template-columns: repeat(auto-fit, minmax(150px, 1fr)); gap: 15px; margin-bottom: 25px; }
        .stat-box { background: #1a1a1a; border-radius: 12px; padding: 20px; text-align: center; border: 1px solid #333; cursor: pointer; transition: all 0.3s; }
        .stat-box:hover { border-color: #ffd700; }
        .stat-box.active { border-color: #ffd700; background: rgba(255,215,0,0.1); }
        .stat-number { font-size: 24px; font-weight: 700; color: #ffd700; }
        .stat-number.green { color: #00ff88; }
        .stat-number.orange { color: #ff9800; }
        .stat-text { font-size: 11px; color: #888; margin-top: 5px; }
        .filters-row { display: flex; gap: 10px; margin-bottom: 20px; flex-wrap: wrap; }
        .filter-btn { padding: 10px 20px; background: #1a1a1a; border: 1px solid #333; border-radius: 8px; color: #888; font-size: 13px; cursor: pointer; text-decoration: none; }
        .filter-btn.active { background: #8b0000; border-color: #8b0000; color: #fff; }
        .requests-grid { display: grid; gap: 15px; }
        .request-card { background: #1a1a1a; border-radius: 15px; padding: 20px; border: 1px solid #333; }
        .request-header { display: flex; justify-content: space-between; align-items: flex-start; margin-bottom: 15px; }
        .request-user { display: flex; align-items: center; gap: 12px; }
        .user-avatar { width: 50px; height: 50px; border-radius: 50%; background: linear-gradient(135deg, #8b0000 0%, #5c0000 100%); display: flex; align-items: center; justify-content: center; font-weight: 700; font-size: 18px; color: #ffd700; }
        .user-info h4 { font-size: 16px; margin-bottom: 3px; }
        .user-info p { font-size: 12px; color: #888; }
        .request-amount { text-align: right; }
        .amount-value { font-size: 24px; font-weight: 800; }
        .amount-value.deposit { color: #00ff88; }
        .amount-value.withdraw { color: #ff9800; }
        .amount-type { font-size: 11px; padding: 4px 10px; border-radius: 10px; text-transform: uppercase; font-weight: 700; margin-top: 5px; display: inline-block; }
        .amount-type.deposit { background: rgba(0,255,136,0.2); color: #00ff88; }
        .amount-type.withdraw { background: rgba(255,152,0,0.2); color: #ff9800; }
        .request-details { display: grid; grid-template-columns: repeat(auto-fit, minmax(120px, 1fr)); gap: 15px; padding: 15px 0; border-top: 1px solid #333; border-bottom: 1px solid #333; margin-bottom: 15px; }
        .detail-item { }
        .detail-label { font-size: 11px; color: #666; margin-bottom: 3px; }
        .detail-value { font-size: 13px; font-weight: 600; }
        .request-actions { display: flex; gap: 10px; flex-wrap: wrap; }
        .btn-action { padding: 10px 20px; border: none; border-radius: 8px; font-size: 13px; font-weight: 600; cursor: pointer; transition: all 0.3s; }
        .btn-action.approve { background: #00c853; color: #fff; }
        .btn-action.reject { background: #333; color: #ff4444; }
        .btn-action.view-ss { background: #333; color: #3498db; }
        .status-badge { padding: 8px 15px; border-radius: 10px; font-size: 12px; font-weight: 700; text-transform: uppercase; }
        .status-badge.approved { background: rgba(0,255,136,0.2); color: #00ff88; }
        .status-badge.rejected { background: rgba(255,68,68,0.2); color: #ff4444; }
        .status-badge.pending { background: rgba(255,215,0,0.2); color: #ffd700; }
        .empty-state { text-align: center; padding: 60px; color: #555; }
        .empty-state .icon { font-size: 50px; margin-bottom: 15px; }
        .modal { display: none; position: fixed; top: 0; left: 0; right: 0; bottom: 0; background: rgba(0,0,0,0.9); z-index: 1000; justify-content: center; align-items: center; padding: 20px; }
        .modal.active { display: flex; }
        .modal-content { background: #1a1a1a; border-radius: 20px; padding: 25px; max-width: 450px; width: 100%; border: 1px solid #333; }
        .modal-header { display: flex; justify-content: space-between; align-items: center; margin-bottom: 20px; }
        .modal-title { font-size: 18px; font-weight: 700; color: #ffd700; }
        .modal-close { background: none; border: none; color: #888; font-size: 24px; cursor: pointer; }
        .form-group { margin-bottom: 20px; }
        .form-label { display: block; font-size: 13px; color: #888; margin-bottom: 8px; }
        .form-select, .form-input, .form-textarea { width: 100%; padding: 14px; background: #252525; border: 2px solid #333; border-radius: 10px; color: #fff; font-size: 15px; outline: none; }
        .form-select:focus, .form-input:focus, .form-textarea:focus { border-color: #ffd700; }
        .form-select option { background: #252525; }
        .form-textarea { min-height: 100px; resize: vertical; }
        .radio-group { display: flex; gap: 15px; }
        .radio-item { flex: 1; padding: 15px; background: #252525; border: 2px solid #333; border-radius: 10px; cursor: pointer; text-align: center; transition: all 0.3s; }
        .radio-item:has(input:checked) { border-color: #ffd700; background: rgba(255,215,0,0.1); }
        .radio-item.credit:has(input:checked) { border-color: #00ff88; background: rgba(0,255,136,0.1); }
        .radio-item.debit:has(input:checked) { border-color: #ff4444; background: rgba(255,68,68,0.1); }
        .radio-item input { display: none; }
        .radio-item label { cursor: pointer; font-weight: 600; }
        .btn-submit { width: 100%; padding: 16px; background: linear-gradient(135deg, #8b0000 0%, #5c0000 100%); border: none; border-radius: 10px; color: #fff; font-size: 16px; font-weight: 700; cursor: pointer; }
        .ss-modal-content { max-width: 600px; }
        .ss-image { width: 100%; border-radius: 10px; margin-top: 15px; }
        .mobile-nav { display: none; position: fixed; bottom: 0; left: 0; right: 0; background: #1a1a1a; padding: 10px 0; border-top: 1px solid #333; z-index: 100; }
        .mobile-nav-inner { display: flex; justify-content: space-around; }
        .mobile-nav-item { text-align: center; color: #666; text-decoration: none; padding: 5px 10px; }
        .mobile-nav-item.active { color: #ffd700; }
        .mobile-nav-icon { font-size: 20px; display: block; margin-bottom: 3px; }
        .mobile-nav-text { font-size: 10px; }
        @media (max-width: 900px) { .sidebar { display: none; } .main-content { margin-left: 0; padding-bottom: 80px; } .mobile-nav { display: block; } }
        @media (max-width: 500px) { .stats-row { grid-template-columns: repeat(2, 1fr); } }
    </style>
</head>
<body>
    <div class="admin-layout">
        <aside class="sidebar">
            <div class="sidebar-header">
                <div class="sidebar-logo">🎰 SATTA KING</div>
                <div class="sidebar-subtitle">Admin Panel</div>
            </div>
            <nav class="sidebar-nav">
                <a href="index.php" class="nav-item"><span class="nav-icon">📊</span><span class="nav-text">Dashboard</span></a>
                <a href="users.php" class="nav-item"><span class="nav-icon">👥</span><span class="nav-text">Users</span><?php if ($pendingUsers > 0): ?><span class="nav-badge"><?= $pendingUsers ?></span><?php endif; ?></a>
                <a href="games.php" class="nav-item"><span class="nav-icon">🎮</span><span class="nav-text">Games</span></a>
                <a href="bids.php" class="nav-item"><span class="nav-icon">🎯</span><span class="nav-text">All Bids</span></a>
                <a href="results.php" class="nav-item"><span class="nav-icon">🏆</span><span class="nav-text">Results</span></a>
                <a href="wallet.php" class="nav-item active"><span class="nav-icon">💰</span><span class="nav-text">Wallet</span><?php if ($pendingRequests > 0): ?><span class="nav-badge"><?= $pendingRequests ?></span><?php endif; ?></a>
                <a href="settings.php" class="nav-item"><span class="nav-icon">⚙️</span><span class="nav-text">Settings</span></a>
                <a href="../logout.php" class="nav-item"><span class="nav-icon">🚪</span><span class="nav-text">Logout</span></a>
            </nav>
        </aside>

        <main class="main-content">
            <div class="top-bar">
                <h1 class="page-title">💰 <span>Wallet Management</span></h1>
                <button class="btn-manual" onclick="openManualModal()">➕ Manual Transaction</button>
            </div>

            <?php if ($success): ?><div class="alert success"><?= $success ?></div><?php endif; ?>
            <?php if ($error): ?><div class="alert error"><?= $error ?></div><?php endif; ?>

            <div class="stats-row">
                <a href="wallet.php?status=pending&type=deposit" class="stat-box <?= $filter_status == 'pending' && $filter_type == 'deposit' ? 'active' : '' ?>">
                    <div class="stat-number orange"><?= $pendingDeposits ?></div>
                    <div class="stat-text">Pending Deposits</div>
                </a>
                <a href="wallet.php?status=pending&type=withdraw" class="stat-box <?= $filter_status == 'pending' && $filter_type == 'withdraw' ? 'active' : '' ?>">
                    <div class="stat-number orange"><?= $pendingWithdraws ?></div>
                    <div class="stat-text">Pending Withdrawals</div>
                </a>
                <div class="stat-box">
                    <div class="stat-number green">₹<?= number_format($todayDeposits, 0) ?></div>
                    <div class="stat-text">Today Deposits</div>
                </div>
                <div class="stat-box">
                    <div class="stat-number" style="color:#ff9800;">₹<?= number_format($todayWithdraws, 0) ?></div>
                    <div class="stat-text">Today Withdrawals</div>
                </div>
            </div>

            <div class="filters-row">
                <a href="wallet.php?status=pending" class="filter-btn <?= $filter_status == 'pending' && !$filter_type ? 'active' : '' ?>">All Pending</a>
                <a href="wallet.php?status=pending&type=deposit" class="filter-btn <?= $filter_status == 'pending' && $filter_type == 'deposit' ? 'active' : '' ?>">⬇️ Deposits</a>
                <a href="wallet.php?status=pending&type=withdraw" class="filter-btn <?= $filter_status == 'pending' && $filter_type == 'withdraw' ? 'active' : '' ?>">⬆️ Withdrawals</a>
                <a href="wallet.php?status=approved" class="filter-btn <?= $filter_status == 'approved' ? 'active' : '' ?>">✅ Approved</a>
                <a href="wallet.php?status=rejected" class="filter-btn <?= $filter_status == 'rejected' ? 'active' : '' ?>">❌ Rejected</a>
                <a href="wallet.php?status=" class="filter-btn <?= !$filter_status ? 'active' : '' ?>">📋 All</a>
            </div>

            <div class="requests-grid">
                <?php if (empty($requests)): ?>
                    <div class="empty-state">
                        <div class="icon">💰</div>
                        <p>No requests found</p>
                    </div>
                <?php else: ?>
                    <?php foreach ($requests as $req): 
                        $userBalance = getWalletBalance($pdo, $req['user_id']);
                    ?>
                        <div class="request-card">
                            <div class="request-header">
                                <div class="request-user">
                                    <div class="user-avatar"><?= strtoupper(substr($req['user_name'], 0, 1)) ?></div>
                                    <div class="user-info">
                                        <h4><?= htmlspecialchars($req['user_name']) ?></h4>
                                        <p>📱 <?= htmlspecialchars($req['mobile']) ?></p>
                                        <p>💰 Balance: ₹<?= number_format($userBalance, 0) ?></p>
                                    </div>
                                </div>
                                <div class="request-amount">
                                    <div class="amount-value <?= $req['type'] ?>"><?= $req['type'] == 'deposit' ? '+' : '-' ?>₹<?= number_format($req['amount'], 0) ?></div>
                                    <div class="amount-type <?= $req['type'] ?>"><?= $req['type'] ?></div>
                                </div>
                            </div>
                            
                            <div class="request-details">
                                <div class="detail-item">
                                    <div class="detail-label">Request ID</div>
                                    <div class="detail-value">#<?= $req['id'] ?></div>
                                </div>
                                <div class="detail-item">
                                    <div class="detail-label">Date</div>
                                    <div class="detail-value"><?= date('d M Y, h:i A', strtotime($req['created_at'])) ?></div>
                                </div>
                                <?php if ($req['type'] == 'withdraw'): ?>
                                    <div class="detail-item">
                                        <div class="detail-label">User UPI</div>
                                        <div class="detail-value" style="color:#ffd700;font-weight:bold;"><?= htmlspecialchars($req['upi_id'] ?? $req['user_upi'] ?? 'Not provided') ?></div>
                                    </div>
                                    <?php if (!empty($req['user_bank'])): ?>
                                    <div class="detail-item">
                                        <div class="detail-label">Bank</div>
                                        <div class="detail-value"><?= htmlspecialchars($req['user_bank']) ?></div>
                                    </div>
                                    <?php endif; ?>
                                    <?php if (!empty($req['user_account'])): ?>
                                    <div class="detail-item">
                                        <div class="detail-label">Account No</div>
                                        <div class="detail-value" style="color:#00ff88;"><?= htmlspecialchars($req['user_account']) ?></div>
                                    </div>
                                    <?php endif; ?>
                                    <?php if (!empty($req['user_account_holder'])): ?>
                                    <div class="detail-item">
                                        <div class="detail-label">Account Holder</div>
                                        <div class="detail-value"><?= htmlspecialchars($req['user_account_holder']) ?></div>
                                    </div>
                                    <?php endif; ?>
                                    <?php if (!empty($req['user_ifsc'])): ?>
                                    <div class="detail-item">
                                        <div class="detail-label">IFSC</div>
                                        <div class="detail-value"><?= htmlspecialchars($req['user_ifsc']) ?></div>
                                    </div>
                                    <?php endif; ?>
                                    <?php if (!empty($req['user_qr'])): ?>
                                    <div class="detail-item" style="grid-column: span 2;">
                                        <div class="detail-label">User QR Code</div>
                                        <div class="detail-value">
                                            <a href="#" onclick="viewScreenshot('../uploads/<?= $req['user_qr'] ?>'); return false;" style="color:#ffd700;">📱 View QR</a>
                                        </div>
                                    </div>
                                    <?php endif; ?>
                                <?php endif; ?>
                                <?php if ($req['type'] == 'deposit' && !empty($req['screenshot'])): ?>
                                    <div class="detail-item">
                                        <div class="detail-label">Screenshot</div>
                                        <div class="detail-value"><a href="#" onclick="viewScreenshot('../uploads/<?= $req['screenshot'] ?>'); return false;">View</a></div>
                                    </div>
                                <?php endif; ?>
                            </div>
                            
                            <?php if ($req['status'] == 'pending'): ?>
                                <div class="request-actions">
                                    <?php if ($req['type'] == 'deposit' && !empty($req['screenshot'])): ?>
                                        <button class="btn-action view-ss" onclick="viewScreenshot('../uploads/<?= $req['screenshot'] ?>')">🖼 Screenshot</button>
                                    <?php endif; ?>
                                    <?php if ($req['type'] == 'withdraw' && !empty($req['user_qr'])): ?>
                                        <button class="btn-action view-ss" onclick="viewScreenshot('../uploads/<?= $req['user_qr'] ?>')" style="background:#9c27b0;">📱 User QR</button>
                                    <?php endif; ?>
                                    <form method="POST" style="display:inline;" onsubmit="return promptCaptcha(this, 'approve')">
                                        <input type="hidden" name="request_id" value="<?= $req['id'] ?>">
                                        <input type="hidden" name="captcha_answer" class="captcha-input">
                                        <button type="submit" name="approve_<?= $req['type'] ?>" class="btn-action approve">✓ Approve</button>
                                    </form>
                                    <button class="btn-action reject" onclick="openRejectModal(<?= $req['id'] ?>)">✕ Reject</button>
                                </div>
                            <?php else: ?>
                                <span class="status-badge <?= $req['status'] ?>"><?= $req['status'] ?></span>
                            <?php endif; ?>
                        </div>
                    <?php endforeach; ?>
                <?php endif; ?>
            </div>
        </main>
    </div>

    <!-- Manual Transaction Modal -->
    <div class="modal" id="manualModal">
        <div class="modal-content">
            <div class="modal-header">
                <div class="modal-title">➕ Manual Transaction</div>
                <button class="modal-close" onclick="closeModal('manualModal')">&times;</button>
            </div>
            <form method="POST">
                <div class="form-group">
                    <label class="form-label">Select User</label>
                    <select name="user_id" class="form-select" required>
                        <option value="">-- Select User --</option>
                        <?php foreach ($users as $u): ?>
                            <option value="<?= $u['id'] ?>"><?= htmlspecialchars($u['name']) ?> (<?= $u['mobile'] ?>)</option>
                        <?php endforeach; ?>
                    </select>
                </div>
                <div class="form-group">
                    <label class="form-label">Transaction Type</label>
                    <div class="radio-group">
                        <div class="radio-item credit">
                            <input type="radio" name="txn_type" id="typeCredit" value="credit" checked>
                            <label for="typeCredit">⬇️ Credit (Add)</label>
                        </div>
                        <div class="radio-item debit">
                            <input type="radio" name="txn_type" id="typeDebit" value="debit">
                            <label for="typeDebit">⬆️ Debit (Deduct)</label>
                        </div>
                    </div>
                </div>
                <div class="form-group">
                    <label class="form-label">Amount (₹)</label>
                    <input type="number" name="amount" class="form-input" placeholder="Enter amount" min="1" required>
                </div>
                <div class="form-group">
                    <label class="form-label">Description</label>
                    <input type="text" name="description" class="form-input" placeholder="e.g. Bonus, Refund, Penalty" required>
                </div>
                <div class="form-group">
                    <label class="form-label">🔢 Captcha: <?= generateAdminCaptcha('manual_txn') ?></label>
                    <input type="number" name="captcha_answer" class="form-input" placeholder="Enter answer" required>
                </div>
                <button type="submit" name="manual_transaction" class="btn-submit">Submit Transaction</button>
            </form>
        </div>
    </div>

    <!-- Reject Modal -->
    <div class="modal" id="rejectModal">
        <div class="modal-content">
            <div class="modal-header">
                <div class="modal-title">❌ Reject Request</div>
                <button class="modal-close" onclick="closeModal('rejectModal')">&times;</button>
            </div>
            <form method="POST">
                <input type="hidden" name="request_id" id="rejectRequestId">
                <div class="form-group">
                    <label class="form-label">Reason (Optional)</label>
                    <textarea name="reason" class="form-textarea" placeholder="Enter rejection reason..."></textarea>
                </div>
                <div class="form-group">
                    <label class="form-label">🔢 Captcha: <?= generateAdminCaptcha('reject') ?></label>
                    <input type="number" name="captcha_answer" class="form-input" placeholder="Enter answer" required>
                </div>
                <button type="submit" name="reject_request" class="btn-submit" style="background: linear-gradient(135deg, #ff4444 0%, #cc0000 100%);">Reject Request</button>
            </form>
        </div>
    </div>

    <!-- Screenshot Modal -->
    <div class="modal" id="ssModal">
        <div class="modal-content ss-modal-content">
            <div class="modal-header">
                <div class="modal-title">🖼 Payment Screenshot</div>
                <button class="modal-close" onclick="closeModal('ssModal')">&times;</button>
            </div>
            <img src="" id="ssImage" class="ss-image" alt="Screenshot">
        </div>
    </div>

    <nav class="mobile-nav">
        <div class="mobile-nav-inner">
            <a href="index.php" class="mobile-nav-item"><span class="mobile-nav-icon">📊</span><span class="mobile-nav-text">Home</span></a>
            <a href="users.php" class="mobile-nav-item"><span class="mobile-nav-icon">👥</span><span class="mobile-nav-text">Users</span></a>
            <a href="results.php" class="mobile-nav-item"><span class="mobile-nav-icon">🏆</span><span class="mobile-nav-text">Results</span></a>
            <a href="wallet.php" class="mobile-nav-item active"><span class="mobile-nav-icon">💰</span><span class="mobile-nav-text">Wallet</span></a>
            <a href="settings.php" class="mobile-nav-item"><span class="mobile-nav-icon">⚙️</span><span class="mobile-nav-text">Settings</span></a>
        </div>
    </nav>

    <script>
        function openManualModal() { document.getElementById('manualModal').classList.add('active'); }
        function openRejectModal(id) { document.getElementById('rejectRequestId').value = id; document.getElementById('rejectModal').classList.add('active'); }
        function viewScreenshot(url) { document.getElementById('ssImage').src = url; document.getElementById('ssModal').classList.add('active'); }
        function closeModal(id) { document.getElementById(id).classList.remove('active'); }
        document.querySelectorAll('.modal').forEach(m => m.addEventListener('click', function(e) { if (e.target === this) this.classList.remove('active'); }));
        
        // Client-side Captcha for inline approve buttons
        function promptCaptcha(form, action) {
            var num1 = Math.floor(Math.random() * 20) + 1;
            var num2 = Math.floor(Math.random() * 20) + 1;
            var correctAnswer = num1 + num2;
            var answer = prompt('🔢 Solve to ' + action + ':\n\n' + num1 + ' + ' + num2 + ' = ?');
            
            if (answer === null) return false;
            
            if (parseInt(answer) !== correctAnswer) {
                alert('❌ Wrong answer! Action cancelled.');
                return false;
            }
            
            // Pass the answer for server-side (backup verification)
            form.querySelector('.captcha-input').value = answer;
            return true;
        }
    </script>
</body>
</html>
