<?php
require_once '../debug.php';
require_once '../config/db.php';
require_once '../config/auth.php';
require_once '../config/functions.php';
require_once '../config/security.php';

initSecureSession();
requireAdmin();

// Check session timeout
if (!checkSessionTimeout($pdo)) {
    header("Location: ../login.php?msg=Session+expired");
    exit();
}

$admin = getUser($pdo);

// Track admin session
trackUserSession($pdo, $admin['id']);

$error = '';
$success = '';

// Activity Log Function with archive names

// Calculation Log Function with archive names

// Handle Publish Result
if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['publish_result'])) {
    if (!verifyAdminCaptcha($_POST['captcha_answer'] ?? '')) {
        $error = getCaptchaError();
    } else {
        $game_id = $_POST['game_id'];
        $playing_date = $_POST['playing_date'];
        $result = str_pad($_POST['result'], 2, '0', STR_PAD_LEFT);
    
    if (empty($game_id) || empty($playing_date) || $result === '') {
        $error = 'All fields are required';
    } elseif (!preg_match('/^[0-9]{2}$/', $result)) {
        $error = 'Result must be 2 digits (00-99)';
    } else {
        $stmt = $pdo->prepare("SELECT * FROM results WHERE game_id = ? AND playing_date = ?");
        $stmt->execute([$game_id, $playing_date]);
        if ($stmt->fetch()) {
            $error = 'Result already published for this game and date!';
        } else {
            $pdo->beginTransaction();
            try {
                // Get game info
                $stmt = $pdo->prepare("SELECT * FROM games WHERE id = ?");
                $stmt->execute([$game_id]);
                $game = $stmt->fetch();
                $rate = $game['rate'];
                
                // Insert result WITH game_name_archive
                $stmt = $pdo->prepare("INSERT INTO results (game_id, game_name_archive, playing_date, result) VALUES (?, ?, ?, ?)");
                $stmt->execute([$game_id, $game['name'], $playing_date, $result]);
                $result_id = $pdo->lastInsertId();
                
                $firstDigit = $result[0];
                $secondDigit = $result[1];
                
                $totalWinners = 0;
                $totalPayout = 0;
                
                // JODI Winners
                $stmt = $pdo->prepare("SELECT b.*, u.name as user_name, u.mobile as user_mobile FROM bids b JOIN users u ON b.user_id = u.id WHERE b.game_id = ? AND b.playing_date = ? AND b.bid_type = 'jodi' AND b.number = ? AND b.status = 'pending'");
                $stmt->execute([$game_id, $playing_date, $result]);
                $jodiWinners = $stmt->fetchAll();
                
                foreach ($jodiWinners as $bid) {
                    $balance_before = getWalletBalance($pdo, $bid['user_id']);
                    $winAmount = $bid['amount'] * $rate;
                    
                    // Update bid
                    $stmt = $pdo->prepare("UPDATE bids SET status = 'won', win_amount = ? WHERE id = ?");
                    $stmt->execute([$winAmount, $bid['id']]);
                    
                    // Add wallet credit WITH archive columns
                    $stmt = $pdo->prepare("INSERT INTO wallet (user_id, user_name_archive, user_mobile_archive, type, amount, description) VALUES (?, ?, ?, 'credit', ?, ?)");
                    $stmt->execute([$bid['user_id'], $bid['user_name'], $bid['user_mobile'], $winAmount, "Won Jodi $result - {$game['name']}"]);
                    
                    $balance_after = getWalletBalance($pdo, $bid['user_id']);
                    
                    // Log calculation
                    logCalculation($pdo, $bid['user_id'], 'WIN_JODI', 'bids', $bid['id'], $winAmount, $balance_before, $balance_after,
                        "Bid Amount ({$bid['amount']}) × Rate ({$rate}) = {$winAmount}",
                        "Jodi win: {$result} on {$game['name']}", $admin['id'], $game['name']
                    );
                    
                    // Add win notification
                    addNotification($pdo, $bid['user_id'], 'win', '🏆 Congratulations! You Won!', 
                        "You won ₹" . number_format($winAmount, 0) . " on Jodi {$result} in {$game['name']}!");
                    
                    $totalWinners++;
                    $totalPayout += $winAmount;
                }
                
                // ANDAR Winners
                $stmt = $pdo->prepare("SELECT b.*, u.name as user_name, u.mobile as user_mobile FROM bids b JOIN users u ON b.user_id = u.id WHERE b.game_id = ? AND b.playing_date = ? AND b.bid_type = 'andar' AND b.number = ? AND b.status = 'pending'");
                $stmt->execute([$game_id, $playing_date, $firstDigit]);
                $andarWinners = $stmt->fetchAll();
                
                foreach ($andarWinners as $bid) {
                    $balance_before = getWalletBalance($pdo, $bid['user_id']);
                    $winAmount = $bid['amount'] * ($rate / 10);
                    
                    $stmt = $pdo->prepare("UPDATE bids SET status = 'won', win_amount = ? WHERE id = ?");
                    $stmt->execute([$winAmount, $bid['id']]);
                    
                    // Add wallet credit WITH archive columns
                    $stmt = $pdo->prepare("INSERT INTO wallet (user_id, user_name_archive, user_mobile_archive, type, amount, description) VALUES (?, ?, ?, 'credit', ?, ?)");
                    $stmt->execute([$bid['user_id'], $bid['user_name'], $bid['user_mobile'], $winAmount, "Won Andar $firstDigit - {$game['name']}"]);
                    
                    $balance_after = getWalletBalance($pdo, $bid['user_id']);
                    
                    logCalculation($pdo, $bid['user_id'], 'WIN_ANDAR', 'bids', $bid['id'], $winAmount, $balance_before, $balance_after,
                        "Bid Amount ({$bid['amount']}) × (Rate/10) ({$rate}/10 = " . ($rate/10) . ") = {$winAmount}",
                        "Andar win: {$firstDigit} on {$game['name']}", $admin['id'], $game['name']
                    );
                    
                    // Add win notification
                    addNotification($pdo, $bid['user_id'], 'win', '🏆 Congratulations! You Won!', 
                        "You won ₹" . number_format($winAmount, 0) . " on Andar {$firstDigit} in {$game['name']}!");
                    
                    $totalWinners++;
                    $totalPayout += $winAmount;
                }
                
                // BAHAR Winners
                $stmt = $pdo->prepare("SELECT b.*, u.name as user_name, u.mobile as user_mobile FROM bids b JOIN users u ON b.user_id = u.id WHERE b.game_id = ? AND b.playing_date = ? AND b.bid_type = 'bahar' AND b.number = ? AND b.status = 'pending'");
                $stmt->execute([$game_id, $playing_date, $secondDigit]);
                $baharWinners = $stmt->fetchAll();
                
                foreach ($baharWinners as $bid) {
                    $balance_before = getWalletBalance($pdo, $bid['user_id']);
                    $winAmount = $bid['amount'] * ($rate / 10);
                    
                    $stmt = $pdo->prepare("UPDATE bids SET status = 'won', win_amount = ? WHERE id = ?");
                    $stmt->execute([$winAmount, $bid['id']]);
                    
                    // Add wallet credit WITH archive columns
                    $stmt = $pdo->prepare("INSERT INTO wallet (user_id, user_name_archive, user_mobile_archive, type, amount, description) VALUES (?, ?, ?, 'credit', ?, ?)");
                    $stmt->execute([$bid['user_id'], $bid['user_name'], $bid['user_mobile'], $winAmount, "Won Bahar $secondDigit - {$game['name']}"]);
                    
                    $balance_after = getWalletBalance($pdo, $bid['user_id']);
                    
                    logCalculation($pdo, $bid['user_id'], 'WIN_BAHAR', 'bids', $bid['id'], $winAmount, $balance_before, $balance_after,
                        "Bid Amount ({$bid['amount']}) × (Rate/10) ({$rate}/10 = " . ($rate/10) . ") = {$winAmount}",
                        "Bahar win: {$secondDigit} on {$game['name']}", $admin['id'], $game['name']
                    );
                    
                    // Add win notification
                    addNotification($pdo, $bid['user_id'], 'win', '🏆 Congratulations! You Won!', 
                        "You won ₹" . number_format($winAmount, 0) . " on Bahar {$secondDigit} in {$game['name']}!");
                    
                    $totalWinners++;
                    $totalPayout += $winAmount;
                }
                
                // Update result stats
                $stmt = $pdo->prepare("UPDATE results SET total_bids = (SELECT COUNT(*) FROM bids WHERE game_id = ? AND playing_date = ?), total_bid_amount = (SELECT COALESCE(SUM(amount), 0) FROM bids WHERE game_id = ? AND playing_date = ?), total_winners = ?, total_payout = ? WHERE id = ?");
                $stmt->execute([$game_id, $playing_date, $game_id, $playing_date, $totalWinners, $totalPayout, $result_id]);
                
                // Mark losers
                $stmt = $pdo->prepare("UPDATE bids SET status = 'lost' WHERE game_id = ? AND playing_date = ? AND status = 'pending'");
                $stmt->execute([$game_id, $playing_date]);
                
                // Log activity
                logActivity($pdo, $admin['id'], 'PUBLISH_RESULT', 'results', $result_id, null, [
                    'game' => $game['name'], 'result' => $result, 'winners' => $totalWinners, 'payout' => $totalPayout
                ], "Result $result published for {$game['name']}. Winners: $totalWinners, Payout: ₹$totalPayout");
                
                $pdo->commit();
                $success = "Result $result published! Winners: $totalWinners, Total Payout: ₹" . number_format($totalPayout, 0);
                
            } catch (Exception $e) {
                $pdo->rollBack();
                $error = 'Error: ' . $e->getMessage();
            }
        }
    }
    }
}

// Handle Delete/Reverse Result
if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['delete_result'])) {
    if (!verifyAdminCaptcha($_POST['captcha_answer'] ?? '')) {
        $error = getCaptchaError();
    } else {
        $result_id = $_POST['result_id'];
    
    // Use LEFT JOIN to handle deleted games
    $stmt = $pdo->prepare("SELECT r.*, COALESCE(g.name, r.game_name_archive, CONCAT('Game #', r.game_id)) as game_name FROM results r LEFT JOIN games g ON r.game_id = g.id WHERE r.id = ?");
    $stmt->execute([$result_id]);
    $resultData = $stmt->fetch();
    
    if ($resultData) {
        $pdo->beginTransaction();
        try {
            // Get all winners with user info
            $stmt = $pdo->prepare("SELECT b.*, u.name as user_name, u.mobile as user_mobile FROM bids b JOIN users u ON b.user_id = u.id WHERE b.game_id = ? AND b.playing_date = ? AND b.status = 'won'");
            $stmt->execute([$resultData['game_id'], $resultData['playing_date']]);
            $winners = $stmt->fetchAll();
            
            $totalReversed = 0;
            
            foreach ($winners as $bid) {
                $balance_before = getWalletBalance($pdo, $bid['user_id']);
                
                // Deduct win amount WITH archive columns
                $stmt = $pdo->prepare("INSERT INTO wallet (user_id, user_name_archive, user_mobile_archive, type, amount, description) VALUES (?, ?, ?, 'debit', ?, ?)");
                $stmt->execute([$bid['user_id'], $bid['user_name'], $bid['user_mobile'], $bid['win_amount'], "Result reversed - {$resultData['game_name']}"]);
                
                $balance_after = getWalletBalance($pdo, $bid['user_id']);
                
                // Log calculation
                logCalculation($pdo, $bid['user_id'], 'RESULT_REVERSE', 'bids', $bid['id'], -$bid['win_amount'], $balance_before, $balance_after,
                    "Win amount reversed: -{$bid['win_amount']}",
                    "Result {$resultData['result']} reversed for {$resultData['game_name']}", $admin['id'], $resultData['game_name']
                );
                
                $totalReversed += $bid['win_amount'];
            }
            
            // Reset bids
            $stmt = $pdo->prepare("UPDATE bids SET status = 'pending', win_amount = 0 WHERE game_id = ? AND playing_date = ?");
            $stmt->execute([$resultData['game_id'], $resultData['playing_date']]);
            
            // Delete result
            $stmt = $pdo->prepare("DELETE FROM results WHERE id = ?");
            $stmt->execute([$result_id]);
            
            // Log activity
            logActivity($pdo, $admin['id'], 'REVERSE_RESULT', 'results', $result_id, [
                'result' => $resultData['result'], 'game' => $resultData['game_name']
            ], null, "Result {$resultData['result']} reversed for {$resultData['game_name']}. Amount reversed: ₹$totalReversed");
            
            $pdo->commit();
            $success = "Result deleted! ₹" . number_format($totalReversed, 0) . " reversed from " . count($winners) . " winners.";
            
        } catch (Exception $e) {
            $pdo->rollBack();
            $error = 'Error: ' . $e->getMessage();
        }
    }
    }
}

// Get games
$stmt = $pdo->query("SELECT * FROM games WHERE status = 'active' ORDER BY name ASC");
$games = $stmt->fetchAll();

// Get recent results with archive support
$stmt = $pdo->query("
    SELECT r.*, 
    COALESCE(g.name, r.game_name_archive, CONCAT('Game #', r.game_id)) as game_name, 
    COALESCE(g.rate, 95) as rate,
    (SELECT COUNT(*) FROM bids WHERE game_id = r.game_id AND playing_date = r.playing_date AND status = 'won') as winners,
    (SELECT COALESCE(SUM(win_amount), 0) FROM bids WHERE game_id = r.game_id AND playing_date = r.playing_date AND status = 'won') as total_payout
    FROM results r 
    LEFT JOIN games g ON r.game_id = g.id 
    ORDER BY r.created_at DESC 
    LIMIT 50
");
$results = $stmt->fetchAll();

// Stats
$stmt = $pdo->query("SELECT COUNT(*) as c FROM results WHERE DATE(created_at) = CURDATE()");
$todayResults = $stmt->fetch()['c'];

$stmt = $pdo->query("SELECT COUNT(*) as c FROM results");
$totalResults = $stmt->fetch()['c'];

$today = date('Y-m-d');
$stmt = $pdo->prepare("SELECT g.* FROM games g WHERE g.status = 'active' AND g.id NOT IN (SELECT game_id FROM results WHERE playing_date = ?) ORDER BY g.close_time ASC");
$stmt->execute([$today]);
$pendingGames = $stmt->fetchAll();

// For sidebar
$stmt = $pdo->query("SELECT COUNT(*) as c FROM users WHERE is_admin = 0 AND status = 'pending'");
$pendingUsers = $stmt->fetch()['c'];
$stmt = $pdo->query("SELECT COUNT(*) as total FROM wallet_requests WHERE status = 'pending'");
$pendingRequests = $stmt->fetch()['total'];
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Results - Admin Panel</title>
    <style>
        * { margin: 0; padding: 0; box-sizing: border-box; }
        body { font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif; background: #0f0f0f; color: #fff; min-height: 100vh; }
        .admin-layout { display: flex; min-height: 100vh; }
        .sidebar { width: 260px; background: #1a1a1a; border-right: 1px solid #333; position: fixed; top: 0; left: 0; bottom: 0; overflow-y: auto; z-index: 100; }
        .sidebar-header { padding: 20px; border-bottom: 1px solid #333; text-align: center; }
        .sidebar-logo { font-size: 22px; font-weight: 800; color: #ffd700; }
        .sidebar-subtitle { font-size: 11px; color: #888; margin-top: 5px; }
        .sidebar-nav { padding: 15px 0; }
        .nav-item { display: flex; align-items: center; gap: 12px; padding: 14px 20px; color: #888; text-decoration: none; transition: all 0.3s; border-left: 3px solid transparent; }
        .nav-item:hover { background: rgba(255,255,255,0.05); color: #fff; }
        .nav-item.active { background: rgba(139,0,0,0.3); color: #ffd700; border-left-color: #ffd700; }
        .nav-icon { font-size: 20px; width: 24px; text-align: center; }
        .nav-text { font-size: 14px; font-weight: 500; }
        .nav-badge { margin-left: auto; background: #ff4444; color: #fff; padding: 2px 8px; border-radius: 10px; font-size: 11px; font-weight: 700; }
        .main-content { flex: 1; margin-left: 260px; padding: 20px; }
        .top-bar { display: flex; justify-content: space-between; align-items: center; margin-bottom: 25px; padding-bottom: 20px; border-bottom: 1px solid #333; }
        .page-title { font-size: 24px; font-weight: 700; }
        .page-title span { color: #ffd700; }
        .alert { padding: 15px; border-radius: 10px; margin-bottom: 20px; font-size: 14px; }
        .alert.success { background: rgba(0,255,0,0.2); border: 1px solid #00cc00; color: #00ff00; }
        .alert.error { background: rgba(255,0,0,0.2); border: 1px solid #ff4444; color: #ff6666; }
        .stats-row { display: grid; grid-template-columns: repeat(auto-fit, minmax(150px, 1fr)); gap: 15px; margin-bottom: 25px; }
        .stat-box { background: #1a1a1a; border-radius: 12px; padding: 20px; text-align: center; border: 1px solid #333; }
        .stat-number { font-size: 28px; font-weight: 700; color: #ffd700; }
        .stat-number.green { color: #00ff88; }
        .stat-number.orange { color: #ff9800; }
        .stat-text { font-size: 12px; color: #888; margin-top: 5px; }
        .content-grid { display: grid; grid-template-columns: 400px 1fr; gap: 25px; }
        .publish-card { background: #1a1a1a; border-radius: 15px; padding: 25px; border: 1px solid #333; height: fit-content; position: sticky; top: 20px; }
        .card-title { font-size: 18px; font-weight: 700; color: #ffd700; margin-bottom: 20px; }
        .form-group { margin-bottom: 20px; }
        .form-label { display: block; font-size: 13px; color: #888; margin-bottom: 8px; }
        .form-select, .form-input { width: 100%; padding: 14px; background: #252525; border: 2px solid #333; border-radius: 10px; color: #fff; font-size: 15px; outline: none; }
        .form-select:focus, .form-input:focus { border-color: #ffd700; }
        .form-select option { background: #252525; }
        .result-input { font-size: 48px; font-weight: 800; text-align: center; letter-spacing: 15px; color: #ffd700; padding: 20px; }
        .result-hint { text-align: center; font-size: 12px; color: #888; margin-top: -10px; margin-bottom: 20px; }
        .btn-publish { width: 100%; padding: 16px; background: linear-gradient(135deg, #00c853 0%, #00a844 100%); border: none; border-radius: 10px; color: #fff; font-size: 16px; font-weight: 700; cursor: pointer; }
        .pending-section { margin-top: 25px; padding-top: 20px; border-top: 1px solid #333; }
        .pending-title { font-size: 14px; color: #ff9800; margin-bottom: 15px; }
        .pending-game { display: flex; justify-content: space-between; padding: 12px 15px; background: #252525; border-radius: 10px; margin-bottom: 10px; cursor: pointer; transition: all 0.3s; }
        .pending-game:hover { background: #333; }
        .results-card { background: #1a1a1a; border-radius: 15px; border: 1px solid #333; overflow: hidden; }
        .results-header { padding: 20px; border-bottom: 1px solid #333; }
        .results-title { font-size: 18px; font-weight: 700; color: #ffd700; }
        .results-list { max-height: 600px; overflow-y: auto; }
        .result-item { display: flex; justify-content: space-between; align-items: center; padding: 18px 20px; border-bottom: 1px solid #333; }
        .result-item:hover { background: #252525; }
        .result-left { display: flex; align-items: center; gap: 15px; }
        .result-number { width: 70px; height: 70px; background: linear-gradient(135deg, #ffd700 0%, #ff8c00 100%); border-radius: 15px; display: flex; align-items: center; justify-content: center; font-size: 28px; font-weight: 800; color: #000; }
        .result-info h4 { font-size: 16px; margin-bottom: 5px; }
        .result-info p { font-size: 12px; color: #888; }
        .result-info .winners { color: #00ff88; margin-top: 5px; }
        .result-right { text-align: right; }
        .result-payout { font-size: 18px; font-weight: 700; color: #ff4444; }
        .result-payout-label { font-size: 11px; color: #888; }
        .btn-delete { padding: 8px 15px; background: rgba(255,68,68,0.2); border: none; border-radius: 6px; color: #ff4444; font-size: 11px; font-weight: 600; cursor: pointer; margin-top: 10px; }
        .modal { display: none; position: fixed; top: 0; left: 0; right: 0; bottom: 0; background: rgba(0,0,0,0.9); z-index: 1000; justify-content: center; align-items: center; padding: 20px; }
        .modal.active { display: flex; }
        .modal-content { background: #1a1a1a; border-radius: 20px; padding: 25px; max-width: 400px; width: 100%; border: 1px solid #333; text-align: center; }
        .modal-icon { font-size: 50px; margin-bottom: 15px; }
        .modal-title { font-size: 20px; font-weight: 700; color: #ff4444; margin-bottom: 10px; }
        .modal-text { font-size: 14px; color: #888; margin-bottom: 20px; }
        .modal-actions { display: flex; gap: 15px; }
        .modal-btn { flex: 1; padding: 14px; border: none; border-radius: 10px; font-size: 14px; font-weight: 700; cursor: pointer; }
        .modal-btn.cancel { background: #333; color: #fff; }
        .modal-btn.confirm { background: #ff4444; color: #fff; }
        .empty-state { text-align: center; padding: 60px; color: #555; }
        .empty-state .icon { font-size: 50px; margin-bottom: 15px; }
        .mobile-nav { display: none; position: fixed; bottom: 0; left: 0; right: 0; background: #1a1a1a; padding: 10px 0; border-top: 1px solid #333; z-index: 100; }
        .mobile-nav-inner { display: flex; justify-content: space-around; }
        .mobile-nav-item { text-align: center; color: #666; text-decoration: none; padding: 5px 10px; }
        .mobile-nav-item.active { color: #ffd700; }
        .mobile-nav-icon { font-size: 20px; display: block; margin-bottom: 3px; }
        .mobile-nav-text { font-size: 10px; }
        @media (max-width: 1100px) { .content-grid { grid-template-columns: 1fr; } .publish-card { position: static; } }
        @media (max-width: 900px) { .sidebar { display: none; } .main-content { margin-left: 0; padding-bottom: 80px; } .mobile-nav { display: block; } }
        @media (max-width: 500px) { .stats-row { grid-template-columns: repeat(2, 1fr); } .result-number { width: 55px; height: 55px; font-size: 22px; } }
    </style>
</head>
<body>
    <div class="admin-layout">
        <aside class="sidebar">
            <div class="sidebar-header">
                <div class="sidebar-logo">🎰 SATTA KING</div>
                <div class="sidebar-subtitle">Admin Panel</div>
            </div>
            <nav class="sidebar-nav">
                <a href="index.php" class="nav-item"><span class="nav-icon">📊</span><span class="nav-text">Dashboard</span></a>
                <a href="users.php" class="nav-item"><span class="nav-icon">👥</span><span class="nav-text">Users</span><?php if ($pendingUsers > 0): ?><span class="nav-badge"><?= $pendingUsers ?></span><?php endif; ?></a>
                <a href="games.php" class="nav-item"><span class="nav-icon">🎮</span><span class="nav-text">Games</span></a>
                <a href="bids.php" class="nav-item"><span class="nav-icon">🎯</span><span class="nav-text">All Bids</span></a>
                <a href="results.php" class="nav-item active"><span class="nav-icon">🏆</span><span class="nav-text">Results</span></a>
                <a href="wallet.php" class="nav-item"><span class="nav-icon">💰</span><span class="nav-text">Wallet</span><?php if ($pendingRequests > 0): ?><span class="nav-badge"><?= $pendingRequests ?></span><?php endif; ?></a>
                <a href="settings.php" class="nav-item"><span class="nav-icon">⚙️</span><span class="nav-text">Settings</span></a>
                <a href="../logout.php" class="nav-item"><span class="nav-icon">🚪</span><span class="nav-text">Logout</span></a>
            </nav>
        </aside>

        <main class="main-content">
            <div class="top-bar">
                <h1 class="page-title">🏆 <span>Results Management</span></h1>
            </div>

            <?php if ($success): ?><div class="alert success"><?= $success ?></div><?php endif; ?>
            <?php if ($error): ?><div class="alert error"><?= $error ?></div><?php endif; ?>

            <div class="stats-row">
                <div class="stat-box"><div class="stat-number"><?= $totalResults ?></div><div class="stat-text">Total Results</div></div>
                <div class="stat-box"><div class="stat-number green"><?= $todayResults ?></div><div class="stat-text">Today Published</div></div>
                <div class="stat-box"><div class="stat-number orange"><?= count($pendingGames) ?></div><div class="stat-text">Pending Today</div></div>
            </div>

            <div class="content-grid">
                <div class="publish-card">
                    <div class="card-title">📢 Publish Result</div>
                    <form method="POST">
                        <div class="form-group">
                            <label class="form-label">Select Game</label>
                            <select name="game_id" class="form-select" required id="gameSelect">
                                <option value="">-- Select Game --</option>
                                <?php foreach ($games as $g): ?>
                                    <option value="<?= $g['id'] ?>"><?= htmlspecialchars($g['name']) ?> (<?= $g['rate'] ?>x)</option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        <div class="form-group">
                            <label class="form-label">Playing Date</label>
                            <input type="date" name="playing_date" class="form-input" value="<?= date('Y-m-d') ?>" required>
                        </div>
                        <div class="form-group">
                            <label class="form-label">Result (00-99)</label>
                            <input type="text" name="result" class="form-input result-input" maxlength="2" placeholder="00" required>
                        </div>
                        <div class="result-hint">Enter 2-digit result number</div>
                        <div class="form-group">
                            <label class="form-label">🔢 Captcha: <?= generateAdminCaptcha('publish') ?></label>
                            <input type="number" name="captcha_answer" class="form-input" placeholder="Enter answer" required>
                        </div>
                        <button type="submit" name="publish_result" class="btn-publish">🏆 PUBLISH RESULT</button>
                    </form>

                    <?php if (!empty($pendingGames)): ?>
                        <div class="pending-section">
                            <div class="pending-title">⏳ Pending Games Today</div>
                            <?php foreach ($pendingGames as $pg): ?>
                                <div class="pending-game" onclick="document.getElementById('gameSelect').value=<?= $pg['id'] ?>">
                                    <span><?= htmlspecialchars($pg['name']) ?></span>
                                    <span style="color:#888"><?= date('h:i A', strtotime($pg['close_time'])) ?></span>
                                </div>
                            <?php endforeach; ?>
                        </div>
                    <?php endif; ?>
                </div>

                <div class="results-card">
                    <div class="results-header"><div class="results-title">📋 Recent Results</div></div>
                    <div class="results-list">
                        <?php if (empty($results)): ?>
                            <div class="empty-state"><div class="icon">🏆</div><p>No results published yet</p></div>
                        <?php else: ?>
                            <?php foreach ($results as $r): ?>
                                <div class="result-item">
                                    <div class="result-left">
                                        <div class="result-number"><?= $r['result'] ?></div>
                                        <div class="result-info">
                                            <h4><?= htmlspecialchars($r['game_name']) ?></h4>
                                            <p>📅 <?= date('d M Y', strtotime($r['playing_date'])) ?></p>
                                            <p class="winners">🏆 <?= $r['winners'] ?> Winners</p>
                                        </div>
                                    </div>
                                    <div class="result-right">
                                        <div class="result-payout">-₹<?= number_format($r['total_payout'], 0) ?></div>
                                        <div class="result-payout-label">Total Payout</div>
                                        <button class="btn-delete" onclick="confirmDelete(<?= $r['id'] ?>, '<?= $r['result'] ?>', '<?= htmlspecialchars(addslashes($r['game_name'])) ?>')">🗑 Delete</button>
                                    </div>
                                </div>
                            <?php endforeach; ?>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </main>
    </div>

    <div class="modal" id="deleteModal">
        <div class="modal-content">
            <div class="modal-icon">⚠️</div>
            <div class="modal-title">Delete Result?</div>
            <div class="modal-text" id="deleteText"></div>
            <form method="POST">
                <input type="hidden" name="result_id" id="deleteResultId">
                <div class="form-group" style="margin:15px 0;">
                    <label class="form-label" style="color:#ffd700;">🔢 Captcha: <?= generateAdminCaptcha('delete_result') ?></label>
                    <input type="number" name="captcha_answer" class="form-input" placeholder="Enter answer" required style="background:#1a1a1a;border:1px solid #333;color:#fff;padding:10px;border-radius:8px;width:100%;margin-top:5px;">
                </div>
                <div class="modal-actions">
                    <button type="button" class="modal-btn cancel" onclick="document.getElementById('deleteModal').classList.remove('active')">Cancel</button>
                    <button type="submit" name="delete_result" class="modal-btn confirm">Delete & Reverse</button>
                </div>
            </form>
        </div>
    </div>

    <nav class="mobile-nav">
        <div class="mobile-nav-inner">
            <a href="index.php" class="mobile-nav-item"><span class="mobile-nav-icon">📊</span><span class="mobile-nav-text">Home</span></a>
            <a href="users.php" class="mobile-nav-item"><span class="mobile-nav-icon">👥</span><span class="mobile-nav-text">Users</span></a>
            <a href="results.php" class="mobile-nav-item active"><span class="mobile-nav-icon">🏆</span><span class="mobile-nav-text">Results</span></a>
            <a href="wallet.php" class="mobile-nav-item"><span class="mobile-nav-icon">💰</span><span class="mobile-nav-text">Wallet</span></a>
            <a href="settings.php" class="mobile-nav-item"><span class="mobile-nav-icon">⚙️</span><span class="mobile-nav-text">Settings</span></a>
        </div>
    </nav>

    <script>
        function confirmDelete(id, result, game) {
            document.getElementById('deleteResultId').value = id;
            document.getElementById('deleteText').innerHTML = `Delete result <strong>${result}</strong> for <strong>${game}</strong>?<br><br>All winnings will be reversed and bids reset to pending.`;
            document.getElementById('deleteModal').classList.add('active');
        }
        document.getElementById('deleteModal').addEventListener('click', function(e) { if (e.target === this) this.classList.remove('active'); });
        document.querySelector('.result-input').addEventListener('blur', function() { if (this.value.length === 1) this.value = '0' + this.value; });
    </script>
</body>
</html>
