<?php
require_once '../debug.php';
require_once '../config/db.php';
require_once '../config/auth.php';
require_once '../config/functions.php';
require_once '../config/security.php';

initSecureSession();
requireAdmin();

// Check session timeout
if (!checkSessionTimeout($pdo)) {
    header("Location: ../login.php?msg=Session+expired");
    exit();
}

$admin = getUser($pdo);

// If admin not found, redirect to login
if (!$admin) {
    session_destroy();
    header("Location: ../login.php");
    exit();
}

// Track admin session
trackUserSession($pdo, $admin['id']);

$error = '';
$success = '';

// Handle User Approve
if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['approve_user'])) {
    $user_id = $_POST['user_id'];
    
    $stmt = $pdo->prepare("SELECT * FROM users WHERE id = ?");
    $stmt->execute([$user_id]);
    $userData = $stmt->fetch();
    
    if ($userData) {
        $stmt = $pdo->prepare("UPDATE users SET status = 'active' WHERE id = ?");
        $stmt->execute([$user_id]);
        
        logActivity($pdo, $admin['id'], 'APPROVE_USER', 'users', $user_id,
            ['status' => 'pending'], ['status' => 'active'],
            "User {$userData['name']} ({$userData['mobile']}) approved from dashboard"
        );
        
        $success = "User {$userData['name']} approved!";
    }
}

// Handle User Reject
if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['reject_user'])) {
    $user_id = $_POST['user_id'];
    
    $stmt = $pdo->prepare("SELECT * FROM users WHERE id = ?");
    $stmt->execute([$user_id]);
    $userData = $stmt->fetch();
    
    if ($userData) {
        $stmt = $pdo->prepare("UPDATE users SET status = 'rejected' WHERE id = ?");
        $stmt->execute([$user_id]);
        
        logActivity($pdo, $admin['id'], 'REJECT_USER', 'users', $user_id,
            ['status' => 'pending'], ['status' => 'rejected'],
            "User {$userData['name']} ({$userData['mobile']}) rejected from dashboard"
        );
        
        $success = "User {$userData['name']} rejected!";
    }
}

// =====================
// STATS QUERIES
// =====================

// Total Users (non-admin)
$stmt = $pdo->query("SELECT COUNT(*) as c FROM users WHERE is_admin = 0");
$totalUsers = $stmt->fetch()['c'] ?? 0;

// Pending Users
$stmt = $pdo->query("SELECT COUNT(*) as c FROM users WHERE is_admin = 0 AND status = 'pending'");
$pendingUsers = $stmt->fetch()['c'] ?? 0;

// Active Games
$stmt = $pdo->query("SELECT COUNT(*) as c FROM games WHERE status = 'active'");
$activeGames = $stmt->fetch()['c'] ?? 0;

// Today's Bids
$stmt = $pdo->query("SELECT COUNT(*) as c, COALESCE(SUM(amount), 0) as amount FROM bids WHERE DATE(created_at) = CURDATE()");
$todayBids = $stmt->fetch();
$todayBids = ['c' => $todayBids['c'] ?? 0, 'amount' => $todayBids['amount'] ?? 0];

// Today's Winnings (Payouts)
$stmt = $pdo->query("SELECT COALESCE(SUM(win_amount), 0) as total FROM bids WHERE status = 'won' AND DATE(created_at) = CURDATE()");
$todayPayouts = $stmt->fetch()['total'] ?? 0;

// Today's Profit/Loss
$todayProfit = $todayBids['amount'] - $todayPayouts;

// Pending Deposits
$stmt = $pdo->query("SELECT COUNT(*) as c, COALESCE(SUM(amount), 0) as amount FROM wallet_requests WHERE type = 'deposit' AND status = 'pending'");
$pendingDeposits = $stmt->fetch();
$pendingDeposits = ['c' => $pendingDeposits['c'] ?? 0, 'amount' => $pendingDeposits['amount'] ?? 0];

// Pending Withdrawals
$stmt = $pdo->query("SELECT COUNT(*) as c, COALESCE(SUM(amount), 0) as amount FROM wallet_requests WHERE type = 'withdraw' AND status = 'pending'");
$pendingWithdraws = $stmt->fetch();
$pendingWithdraws = ['c' => $pendingWithdraws['c'] ?? 0, 'amount' => $pendingWithdraws['amount'] ?? 0];

// Total Wallet Balance (All Users)
$stmt = $pdo->query("SELECT 
    COALESCE(SUM(CASE WHEN type = 'credit' THEN amount ELSE 0 END), 0) -
    COALESCE(SUM(CASE WHEN type = 'debit' THEN amount ELSE 0 END), 0) as total
    FROM wallet");
$totalWalletBalance = $stmt->fetch()['total'] ?? 0;

// =====================
// RECENT DATA
// =====================

// Pending Users (latest 5)
$stmt = $pdo->query("SELECT * FROM users WHERE is_admin = 0 AND status = 'pending' ORDER BY created_at DESC LIMIT 5");
$pendingUsersList = $stmt->fetchAll() ?: [];

// Pending Wallet Requests (latest 5) - with LEFT JOIN for archive support
$stmt = $pdo->query("
    SELECT wr.*, 
    COALESCE(u.name, wr.user_name_archive, CONCAT('User #', wr.user_id)) as user_name, 
    COALESCE(u.mobile, wr.user_mobile_archive) as mobile 
    FROM wallet_requests wr 
    LEFT JOIN users u ON wr.user_id = u.id 
    WHERE wr.status = 'pending' 
    ORDER BY wr.created_at DESC 
    LIMIT 5
");
$pendingRequests = $stmt->fetchAll() ?: [];

// Recent Bids (latest 10) - With Archive Support
$stmt = $pdo->query("
    SELECT b.*, 
    COALESCE(u.name, b.user_name_archive, CONCAT('User #', b.user_id)) as user_name, 
    COALESCE(u.mobile, b.user_mobile_archive) as mobile,
    COALESCE(g.name, b.game_name_archive, CONCAT('Game #', b.game_id)) as game_name
    FROM bids b 
    LEFT JOIN users u ON b.user_id = u.id 
    LEFT JOIN games g ON b.game_id = g.id 
    ORDER BY b.created_at DESC 
    LIMIT 10
");
$recentBids = $stmt->fetchAll() ?: [];

// Today's Results - With Archive Support
$today = date('Y-m-d');
$stmt = $pdo->prepare("
    SELECT r.*, 
    COALESCE(g.name, r.game_name_archive, CONCAT('Game #', r.game_id)) as game_name
    FROM results r 
    LEFT JOIN games g ON r.game_id = g.id 
    WHERE r.playing_date = ? 
    ORDER BY r.created_at DESC
");
$stmt->execute([$today]);
$todayResults = $stmt->fetchAll() ?: [];

// Recent Activity Log (latest 10)
$recentActivities = [];
try {
    $stmt = $pdo->query("
        SELECT al.*, 
        COALESCE(al.admin_name_archive, u.name, a.name, 'System') as performer_name
        FROM activity_log al 
        LEFT JOIN users u ON al.user_id = u.id 
        LEFT JOIN users a ON al.admin_id = a.id 
        ORDER BY al.created_at DESC 
        LIMIT 10
    ");
    $recentActivities = $stmt->fetchAll() ?: [];
} catch (Exception $e) {
    // Table might not exist yet
}

// For sidebar badges
$pendingWalletTotal = $pendingDeposits['c'] + $pendingWithdraws['c'];
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Dashboard - Admin Panel</title>
    <style>
        * { margin: 0; padding: 0; box-sizing: border-box; }
        body { font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif; background: #0f0f0f; color: #fff; min-height: 100vh; }
        
        .admin-layout { display: flex; min-height: 100vh; }
        
        /* Sidebar */
        .sidebar { width: 260px; background: #1a1a1a; border-right: 1px solid #333; position: fixed; top: 0; left: 0; bottom: 0; overflow-y: auto; z-index: 100; }
        .sidebar-header { padding: 20px; border-bottom: 1px solid #333; text-align: center; }
        .sidebar-logo { font-size: 22px; font-weight: 800; color: #ffd700; }
        .sidebar-subtitle { font-size: 11px; color: #888; margin-top: 5px; }
        .sidebar-nav { padding: 15px 0; }
        .nav-item { display: flex; align-items: center; gap: 12px; padding: 14px 20px; color: #888; text-decoration: none; transition: all 0.3s; border-left: 3px solid transparent; }
        .nav-item:hover { background: rgba(255,255,255,0.05); color: #fff; }
        .nav-item.active { background: rgba(139,0,0,0.3); color: #ffd700; border-left-color: #ffd700; }
        .nav-icon { font-size: 20px; width: 24px; text-align: center; }
        .nav-text { font-size: 14px; font-weight: 500; }
        .nav-badge { margin-left: auto; background: #ff4444; color: #fff; padding: 2px 8px; border-radius: 10px; font-size: 11px; font-weight: 700; }
        
        /* Main Content */
        .main-content { flex: 1; margin-left: 260px; padding: 20px; }
        
        .top-bar { display: flex; justify-content: space-between; align-items: center; margin-bottom: 25px; padding-bottom: 20px; border-bottom: 1px solid #333; flex-wrap: wrap; gap: 15px; }
        .page-title { font-size: 24px; font-weight: 700; }
        .page-title span { color: #ffd700; }
        .admin-info { display: flex; align-items: center; gap: 12px; }
        .admin-info span { font-size: 14px; color: #888; }
        .admin-avatar { width: 40px; height: 40px; border-radius: 50%; background: linear-gradient(135deg, #ffd700 0%, #ff8c00 100%); display: flex; align-items: center; justify-content: center; font-weight: 700; color: #000; }
        
        .alert { padding: 15px; border-radius: 10px; margin-bottom: 20px; font-size: 14px; }
        .alert.success { background: rgba(0,255,0,0.2); border: 1px solid #00cc00; color: #00ff00; }
        .alert.error { background: rgba(255,0,0,0.2); border: 1px solid #ff4444; color: #ff6666; }
        
        /* Stats Grid */
        .stats-grid { display: grid; grid-template-columns: repeat(auto-fit, minmax(180px, 1fr)); gap: 20px; margin-bottom: 30px; }
        .stat-card { background: #1a1a1a; border-radius: 15px; padding: 20px; border: 1px solid #333; transition: all 0.3s; }
        .stat-card.clickable { cursor: pointer; }
        .stat-card.clickable:hover { border-color: #ffd700; transform: translateY(-2px); }
        .stat-header { display: flex; justify-content: space-between; align-items: center; margin-bottom: 15px; }
        .stat-icon { width: 45px; height: 45px; border-radius: 12px; display: flex; align-items: center; justify-content: center; font-size: 22px; }
        .stat-icon.blue { background: rgba(52,152,219,0.2); }
        .stat-icon.orange { background: rgba(255,152,0,0.2); }
        .stat-icon.green { background: rgba(0,255,136,0.2); }
        .stat-icon.purple { background: rgba(155,89,182,0.2); }
        .stat-icon.red { background: rgba(255,68,68,0.2); }
        .stat-icon.gold { background: rgba(255,215,0,0.2); }
        .stat-trend { font-size: 11px; padding: 4px 8px; border-radius: 10px; }
        .stat-trend.up { background: rgba(0,255,136,0.2); color: #00ff88; }
        .stat-trend.down { background: rgba(255,152,0,0.2); color: #ff9800; }
        .stat-value { font-size: 28px; font-weight: 800; margin-bottom: 5px; }
        .stat-value.green { color: #00ff88; }
        .stat-value.red { color: #ff4444; }
        .stat-value.gold { color: #ffd700; }
        .stat-label { font-size: 12px; color: #888; }
        
        /* Content Grid */
        .content-grid { display: grid; grid-template-columns: repeat(auto-fit, minmax(350px, 1fr)); gap: 25px; margin-bottom: 30px; }
        .content-card { background: #1a1a1a; border-radius: 15px; border: 1px solid #333; overflow: hidden; }
        .card-header { padding: 18px 20px; border-bottom: 1px solid #333; display: flex; justify-content: space-between; align-items: center; }
        .card-title { font-size: 16px; font-weight: 700; }
        .card-title span { color: #ffd700; }
        .card-link { font-size: 12px; color: #ffd700; text-decoration: none; }
        .card-content { padding: 15px; max-height: 350px; overflow-y: auto; }
        
        /* Pending Users */
        .pending-user { display: flex; justify-content: space-between; align-items: center; padding: 12px 15px; background: #252525; border-radius: 10px; margin-bottom: 10px; }
        .pending-user:last-child { margin-bottom: 0; }
        .user-info { display: flex; align-items: center; gap: 12px; }
        .user-avatar { width: 40px; height: 40px; border-radius: 50%; background: linear-gradient(135deg, #8b0000 0%, #5c0000 100%); display: flex; align-items: center; justify-content: center; font-weight: 700; color: #ffd700; }
        .user-details h4 { font-size: 14px; margin-bottom: 2px; }
        .user-details p { font-size: 11px; color: #888; }
        .user-actions { display: flex; gap: 8px; }
        .btn-approve { padding: 8px 12px; background: #00c853; border: none; border-radius: 6px; color: #fff; font-size: 12px; font-weight: 600; cursor: pointer; }
        .btn-reject { padding: 8px 12px; background: #333; border: none; border-radius: 6px; color: #ff4444; font-size: 12px; font-weight: 600; cursor: pointer; }
        
        /* Wallet Requests */
        .wallet-request { display: flex; justify-content: space-between; align-items: center; padding: 12px 15px; background: #252525; border-radius: 10px; margin-bottom: 10px; }
        .wallet-request:last-child { margin-bottom: 0; }
        .request-info h4 { font-size: 14px; margin-bottom: 2px; }
        .request-info p { font-size: 11px; color: #888; }
        .request-amount { text-align: right; }
        .request-amount .amount { font-size: 16px; font-weight: 700; }
        .request-amount .amount.deposit { color: #00ff88; }
        .request-amount .amount.withdraw { color: #ff9800; }
        .request-amount .type { font-size: 10px; color: #888; }
        
        /* Recent Bids */
        .bid-item { display: flex; justify-content: space-between; align-items: center; padding: 12px 15px; background: #252525; border-radius: 10px; margin-bottom: 10px; }
        .bid-item:last-child { margin-bottom: 0; }
        .bid-info { }
        .bid-info h4 { font-size: 14px; margin-bottom: 2px; }
        .bid-info p { font-size: 11px; color: #888; }
        .bid-details { text-align: right; }
        .bid-number { font-size: 18px; font-weight: 800; color: #ffd700; margin-bottom: 2px; }
        .bid-amount { font-size: 12px; color: #888; }
        .bid-status { display: inline-block; padding: 3px 8px; border-radius: 10px; font-size: 9px; font-weight: 700; text-transform: uppercase; margin-top: 5px; }
        .bid-status.pending { background: rgba(255,215,0,0.2); color: #ffd700; }
        .bid-status.won { background: rgba(0,255,136,0.2); color: #00ff88; }
        .bid-status.lost { background: rgba(255,68,68,0.2); color: #ff4444; }
        
        /* Today's Results */
        .result-item { display: flex; align-items: center; gap: 15px; padding: 12px 15px; background: #252525; border-radius: 10px; margin-bottom: 10px; }
        .result-item:last-child { margin-bottom: 0; }
        .result-number { width: 50px; height: 50px; background: linear-gradient(135deg, #ffd700 0%, #ff8c00 100%); border-radius: 12px; display: flex; align-items: center; justify-content: center; font-size: 22px; font-weight: 800; color: #000; }
        .result-info h4 { font-size: 14px; margin-bottom: 2px; }
        .result-info p { font-size: 11px; color: #888; }
        
        /* Activity Log */
        .activity-item { padding: 12px 15px; border-bottom: 1px solid #333; }
        .activity-item:last-child { border-bottom: none; }
        .activity-text { font-size: 13px; margin-bottom: 5px; }
        .activity-meta { font-size: 11px; color: #666; }
        
        /* Empty State */
        .empty-state { text-align: center; padding: 30px; color: #555; }
        .empty-state .icon { font-size: 30px; margin-bottom: 10px; }
        .empty-state p { font-size: 13px; }
        
        /* Mobile Nav */
        .mobile-nav { display: none; position: fixed; bottom: 0; left: 0; right: 0; background: #1a1a1a; padding: 10px 0; border-top: 1px solid #333; z-index: 100; }
        .mobile-nav-inner { display: flex; justify-content: space-around; }
        .mobile-nav-item { text-align: center; color: #666; text-decoration: none; padding: 5px 10px; }
        .mobile-nav-item.active { color: #ffd700; }
        .mobile-nav-icon { font-size: 20px; display: block; margin-bottom: 3px; }
        .mobile-nav-text { font-size: 10px; }
        
        @media (max-width: 900px) {
            .sidebar { display: none; }
            .main-content { margin-left: 0; padding-bottom: 80px; }
            .mobile-nav { display: block; }
        }
        @media (max-width: 500px) {
            .stats-grid { grid-template-columns: repeat(2, 1fr); }
            .content-grid { grid-template-columns: 1fr; }
        }
    </style>
</head>
<body>
    <div class="admin-layout">
        <aside class="sidebar">
            <div class="sidebar-header">
                <div class="sidebar-logo">🎰 SATTA KING</div>
                <div class="sidebar-subtitle">Admin Panel</div>
            </div>
            <nav class="sidebar-nav">
                <a href="index.php" class="nav-item active"><span class="nav-icon">📊</span><span class="nav-text">Dashboard</span></a>
                <a href="users.php" class="nav-item"><span class="nav-icon">👥</span><span class="nav-text">Users</span><?php if ($pendingUsers > 0): ?><span class="nav-badge"><?= $pendingUsers ?></span><?php endif; ?></a>
                <a href="games.php" class="nav-item"><span class="nav-icon">🎮</span><span class="nav-text">Games</span></a>
                <a href="bids.php" class="nav-item"><span class="nav-icon">🎯</span><span class="nav-text">All Bids</span></a>
                <a href="results.php" class="nav-item"><span class="nav-icon">🏆</span><span class="nav-text">Results</span></a>
                <a href="wallet.php" class="nav-item"><span class="nav-icon">💰</span><span class="nav-text">Wallet</span><?php if ($pendingWalletTotal > 0): ?><span class="nav-badge"><?= $pendingWalletTotal ?></span><?php endif; ?></a>
                <a href="settings.php" class="nav-item"><span class="nav-icon">⚙️</span><span class="nav-text">Settings</span></a>
                <a href="../logout.php" class="nav-item"><span class="nav-icon">🚪</span><span class="nav-text">Logout</span></a>
            </nav>
        </aside>

        <main class="main-content">
            <div class="top-bar">
                <h1 class="page-title">📊 <span>Dashboard</span></h1>
                <div class="admin-info">
                    <span>Welcome, <?= htmlspecialchars($admin['name'] ?? 'Admin') ?></span>
                    <div class="admin-avatar"><?= strtoupper(substr($admin['name'] ?? 'A', 0, 1)) ?></div>
                </div>
            </div>

            <?php if ($success): ?><div class="alert success"><?= $success ?></div><?php endif; ?>
            <?php if ($error): ?><div class="alert error"><?= $error ?></div><?php endif; ?>

            <!-- Stats Grid -->
            <div class="stats-grid">
                <div class="stat-card clickable" onclick="location.href='users.php'">
                    <div class="stat-header">
                        <div class="stat-icon blue">👥</div>
                    </div>
                    <div class="stat-value"><?= number_format($totalUsers) ?></div>
                    <div class="stat-label">Total Users</div>
                </div>
                
                <div class="stat-card clickable" onclick="location.href='users.php?filter=pending'">
                    <div class="stat-header">
                        <div class="stat-icon orange">⏳</div>
                        <?php if ($pendingUsers > 0): ?><span class="stat-trend down"><?= $pendingUsers ?> pending</span><?php endif; ?>
                    </div>
                    <div class="stat-value <?= $pendingUsers > 0 ? 'gold' : '' ?>"><?= $pendingUsers ?></div>
                    <div class="stat-label">Pending Approvals</div>
                </div>
                
                <div class="stat-card clickable" onclick="location.href='games.php'">
                    <div class="stat-header">
                        <div class="stat-icon green">🎮</div>
                    </div>
                    <div class="stat-value"><?= $activeGames ?></div>
                    <div class="stat-label">Active Games</div>
                </div>
                
                <div class="stat-card clickable" onclick="location.href='bids.php'">
                    <div class="stat-header">
                        <div class="stat-icon purple">🎯</div>
                        <span class="stat-trend up"><?= $todayBids['c'] ?> today</span>
                    </div>
                    <div class="stat-value">₹<?= number_format($todayBids['amount'], 0) ?></div>
                    <div class="stat-label">Today's Bids</div>
                </div>
                
                <div class="stat-card">
                    <div class="stat-header">
                        <div class="stat-icon red">💸</div>
                    </div>
                    <div class="stat-value red">₹<?= number_format($todayPayouts, 0) ?></div>
                    <div class="stat-label">Today's Payouts</div>
                </div>
                
                <div class="stat-card">
                    <div class="stat-header">
                        <div class="stat-icon gold">📈</div>
                    </div>
                    <div class="stat-value <?= $todayProfit >= 0 ? 'green' : 'red' ?>">
                        <?= $todayProfit >= 0 ? '+' : '' ?>₹<?= number_format($todayProfit, 0) ?>
                    </div>
                    <div class="stat-label">Today's Profit/Loss</div>
                </div>
                
                <div class="stat-card clickable" onclick="location.href='wallet.php?status=pending&type=deposit'">
                    <div class="stat-header">
                        <div class="stat-icon green">⬇️</div>
                        <?php if ($pendingDeposits['c'] > 0): ?><span class="stat-trend down"><?= $pendingDeposits['c'] ?> pending</span><?php endif; ?>
                    </div>
                    <div class="stat-value green">₹<?= number_format($pendingDeposits['amount'], 0) ?></div>
                    <div class="stat-label">Pending Deposits</div>
                </div>
                
                <div class="stat-card clickable" onclick="location.href='wallet.php?status=pending&type=withdraw'">
                    <div class="stat-header">
                        <div class="stat-icon orange">⬆️</div>
                        <?php if ($pendingWithdraws['c'] > 0): ?><span class="stat-trend down"><?= $pendingWithdraws['c'] ?> pending</span><?php endif; ?>
                    </div>
                    <div class="stat-value" style="color:#ff9800;">₹<?= number_format($pendingWithdraws['amount'], 0) ?></div>
                    <div class="stat-label">Pending Withdrawals</div>
                </div>
            </div>

            <!-- Content Grid -->
            <div class="content-grid">
                <!-- Pending Users -->
                <div class="content-card">
                    <div class="card-header">
                        <div class="card-title">⏳ <span>Pending Users</span></div>
                        <a href="users.php?filter=pending" class="card-link">View All →</a>
                    </div>
                    <div class="card-content">
                        <?php if (empty($pendingUsersList)): ?>
                            <div class="empty-state"><div class="icon">✅</div><p>No pending users</p></div>
                        <?php else: ?>
                            <?php foreach ($pendingUsersList as $u): ?>
                                <div class="pending-user">
                                    <div class="user-info">
                                        <div class="user-avatar"><?= strtoupper(substr($u['name'] ?? 'U', 0, 1)) ?></div>
                                        <div class="user-details">
                                            <h4><?= htmlspecialchars($u['name'] ?? 'Unknown') ?></h4>
                                            <p><?= htmlspecialchars($u['mobile'] ?? '') ?> • <?= date('d M', strtotime($u['created_at'])) ?></p>
                                        </div>
                                    </div>
                                    <div class="user-actions">
                                        <form method="POST" style="display:inline;">
                                            <input type="hidden" name="user_id" value="<?= $u['id'] ?>">
                                            <button type="submit" name="approve_user" class="btn-approve">✓</button>
                                        </form>
                                        <form method="POST" style="display:inline;">
                                            <input type="hidden" name="user_id" value="<?= $u['id'] ?>">
                                            <button type="submit" name="reject_user" class="btn-reject">✕</button>
                                        </form>
                                    </div>
                                </div>
                            <?php endforeach; ?>
                        <?php endif; ?>
                    </div>
                </div>

                <!-- Pending Wallet Requests -->
                <div class="content-card">
                    <div class="card-header">
                        <div class="card-title">💰 <span>Wallet Requests</span></div>
                        <a href="wallet.php?status=pending" class="card-link">View All →</a>
                    </div>
                    <div class="card-content">
                        <?php if (empty($pendingRequests)): ?>
                            <div class="empty-state"><div class="icon">✅</div><p>No pending requests</p></div>
                        <?php else: ?>
                            <?php foreach ($pendingRequests as $req): ?>
                                <div class="wallet-request">
                                    <div class="request-info">
                                        <h4><?= htmlspecialchars($req['user_name'] ?? 'Unknown') ?></h4>
                                        <p><?= htmlspecialchars($req['mobile'] ?? '') ?> • <?= date('d M, h:i A', strtotime($req['created_at'])) ?></p>
                                    </div>
                                    <div class="request-amount">
                                        <div class="amount <?= $req['type'] ?>"><?= $req['type'] == 'deposit' ? '+' : '-' ?>₹<?= number_format($req['amount'], 0) ?></div>
                                        <div class="type"><?= ucfirst($req['type']) ?></div>
                                    </div>
                                </div>
                            <?php endforeach; ?>
                        <?php endif; ?>
                    </div>
                </div>

                <!-- Recent Bids -->
                <div class="content-card">
                    <div class="card-header">
                        <div class="card-title">🎯 <span>Recent Bids</span></div>
                        <a href="bids.php" class="card-link">View All →</a>
                    </div>
                    <div class="card-content">
                        <?php if (empty($recentBids)): ?>
                            <div class="empty-state"><div class="icon">🎯</div><p>No bids yet</p></div>
                        <?php else: ?>
                            <?php foreach ($recentBids as $bid): ?>
                                <div class="bid-item">
                                    <div class="bid-info">
                                        <h4><?= htmlspecialchars($bid['user_name'] ?? 'Unknown') ?></h4>
                                        <p><?= htmlspecialchars($bid['game_name'] ?? 'Unknown Game') ?> • <?= ucfirst($bid['bid_type']) ?></p>
                                    </div>
                                    <div class="bid-details">
                                        <div class="bid-number"><?= $bid['number'] ?></div>
                                        <div class="bid-amount">₹<?= number_format($bid['amount'], 0) ?></div>
                                        <span class="bid-status <?= $bid['status'] ?>"><?= $bid['status'] ?></span>
                                    </div>
                                </div>
                            <?php endforeach; ?>
                        <?php endif; ?>
                    </div>
                </div>

                <!-- Today's Results -->
                <div class="content-card">
                    <div class="card-header">
                        <div class="card-title">🏆 <span>Today's Results</span></div>
                        <a href="results.php" class="card-link">Publish →</a>
                    </div>
                    <div class="card-content">
                        <?php if (empty($todayResults)): ?>
                            <div class="empty-state"><div class="icon">🏆</div><p>No results today</p></div>
                        <?php else: ?>
                            <?php foreach ($todayResults as $r): ?>
                                <div class="result-item">
                                    <div class="result-number"><?= $r['result'] ?></div>
                                    <div class="result-info">
                                        <h4><?= htmlspecialchars($r['game_name'] ?? 'Unknown Game') ?></h4>
                                        <p>Published <?= date('h:i A', strtotime($r['created_at'])) ?></p>
                                    </div>
                                </div>
                            <?php endforeach; ?>
                        <?php endif; ?>
                    </div>
                </div>
            </div>

            <!-- Activity Log -->
            <div class="content-card">
                <div class="card-header">
                    <div class="card-title">📋 <span>Recent Activity</span></div>
                </div>
                <div class="card-content">
                    <?php if (empty($recentActivities)): ?>
                        <div class="empty-state"><div class="icon">📋</div><p>No recent activity</p></div>
                    <?php else: ?>
                        <?php foreach ($recentActivities as $activity): ?>
                            <div class="activity-item">
                                <div class="activity-text"><?= htmlspecialchars($activity['description'] ?? 'Activity') ?></div>
                                <div class="activity-meta">
                                    By <?= htmlspecialchars($activity['performer_name'] ?? 'System') ?> • 
                                    <?= date('d M Y, h:i A', strtotime($activity['created_at'])) ?>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    <?php endif; ?>
                </div>
            </div>
        </main>
    </div>

    <nav class="mobile-nav">
        <div class="mobile-nav-inner">
            <a href="index.php" class="mobile-nav-item active"><span class="mobile-nav-icon">📊</span><span class="mobile-nav-text">Home</span></a>
            <a href="users.php" class="mobile-nav-item"><span class="mobile-nav-icon">👥</span><span class="mobile-nav-text">Users</span></a>
            <a href="results.php" class="mobile-nav-item"><span class="mobile-nav-icon">🏆</span><span class="mobile-nav-text">Results</span></a>
            <a href="wallet.php" class="mobile-nav-item"><span class="mobile-nav-icon">💰</span><span class="mobile-nav-text">Wallet</span></a>
            <a href="settings.php" class="mobile-nav-item"><span class="mobile-nav-icon">⚙️</span><span class="mobile-nav-text">Settings</span></a>
        </div>
    </nav>
</body>
</html>