<?php
require_once '../debug.php';
require_once '../config/db.php';
require_once '../config/auth.php';
require_once '../config/functions.php';
require_once '../config/security.php';

initSecureSession();
requireAdmin();

if (!checkSessionTimeout($pdo)) {
    header("Location: ../login.php?msg=Session+expired");
    exit();
}

$admin = getUser($pdo);
trackUserSession($pdo, $admin['id']);

// Filters
$filter_game = $_GET['game'] ?? '';
$filter_date = $_GET['date'] ?? '';
$filter_status = $_GET['status'] ?? '';
$filter_type = $_GET['type'] ?? '';
$search = $_GET['search'] ?? '';

$where = "WHERE 1=1";
$params = [];

if ($filter_game) {
    $where .= " AND b.game_id = ?";
    $params[] = $filter_game;
}
if ($filter_date) {
    $where .= " AND b.playing_date = ?";
    $params[] = $filter_date;
}
if ($filter_status) {
    $where .= " AND b.status = ?";
    $params[] = $filter_status;
}
if ($filter_type) {
    $where .= " AND b.bid_type = ?";
    $params[] = $filter_type;
}
if ($search) {
    $where .= " AND (u.name LIKE ? OR u.mobile LIKE ? OR b.number LIKE ?)";
    $params[] = "%$search%";
    $params[] = "%$search%";
    $params[] = "%$search%";
}

// Get bids with LEFT JOIN to handle deleted games
$sql = "SELECT b.*, u.name as user_name, u.mobile, 
        COALESCE(g.name, b.game_name_archive, CONCAT('Game #', b.game_id, ' (Deleted)')) as game_name,
        g.rate as current_rate
        FROM bids b 
        JOIN users u ON b.user_id = u.id 
        LEFT JOIN games g ON b.game_id = g.id 
        $where 
        ORDER BY b.created_at DESC 
        LIMIT 500";
$stmt = $pdo->prepare($sql);
$stmt->execute($params);
$bids = $stmt->fetchAll();

// Get all games (active + archived for filter)
$stmt = $pdo->query("SELECT id, name FROM games ORDER BY name ASC");
$games = $stmt->fetchAll();

// Also get archived games for filter dropdown
$stmt = $pdo->query("SELECT id, name FROM games_archive ORDER BY name ASC");
$archivedGames = $stmt->fetchAll();

// Stats
$stmt = $pdo->query("SELECT COUNT(*) as c FROM bids");
$totalBids = $stmt->fetch()['c'];

$stmt = $pdo->query("SELECT COUNT(*) as c FROM bids WHERE DATE(created_at) = CURDATE()");
$todayBids = $stmt->fetch()['c'];

$stmt = $pdo->query("SELECT COALESCE(SUM(amount), 0) as total FROM bids");
$totalBidAmount = $stmt->fetch()['total'];

$stmt = $pdo->query("SELECT COALESCE(SUM(win_amount), 0) as total FROM bids WHERE status = 'won'");
$totalPayouts = $stmt->fetch()['total'];

$profit = $totalBidAmount - $totalPayouts;

// For sidebar
$stmt = $pdo->query("SELECT COUNT(*) as c FROM users WHERE is_admin = 0 AND status = 'pending'");
$pendingUsers = $stmt->fetch()['c'];
$stmt = $pdo->query("SELECT COUNT(*) as total FROM wallet_requests WHERE status = 'pending'");
$pendingRequests = $stmt->fetch()['total'];
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>All Bids - Admin Panel</title>
    <style>
        * { margin: 0; padding: 0; box-sizing: border-box; }
        body { font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif; background: #0f0f0f; color: #fff; min-height: 100vh; }
        
        .admin-layout { display: flex; min-height: 100vh; }
        
        .sidebar { width: 260px; background: #1a1a1a; border-right: 1px solid #333; position: fixed; top: 0; left: 0; bottom: 0; overflow-y: auto; z-index: 100; }
        .sidebar-header { padding: 20px; border-bottom: 1px solid #333; text-align: center; }
        .sidebar-logo { font-size: 22px; font-weight: 800; color: #ffd700; }
        .sidebar-subtitle { font-size: 11px; color: #888; margin-top: 5px; }
        .sidebar-nav { padding: 15px 0; }
        .nav-item { display: flex; align-items: center; gap: 12px; padding: 14px 20px; color: #888; text-decoration: none; transition: all 0.3s; border-left: 3px solid transparent; }
        .nav-item:hover { background: rgba(255,255,255,0.05); color: #fff; }
        .nav-item.active { background: rgba(139,0,0,0.3); color: #ffd700; border-left-color: #ffd700; }
        .nav-icon { font-size: 20px; width: 24px; text-align: center; }
        .nav-text { font-size: 14px; font-weight: 500; }
        .nav-badge { margin-left: auto; background: #ff4444; color: #fff; padding: 2px 8px; border-radius: 10px; font-size: 11px; font-weight: 700; }
        
        .main-content { flex: 1; margin-left: 260px; padding: 20px; }
        
        .top-bar { display: flex; justify-content: space-between; align-items: center; margin-bottom: 25px; padding-bottom: 20px; border-bottom: 1px solid #333; }
        .page-title { font-size: 24px; font-weight: 700; }
        .page-title span { color: #ffd700; }
        
        /* Stats */
        .stats-row { display: grid; grid-template-columns: repeat(auto-fit, minmax(150px, 1fr)); gap: 15px; margin-bottom: 25px; }
        .stat-box { background: linear-gradient(145deg, #1e1e1e 0%, #2a2a2a 100%); border-radius: 15px; padding: 20px; border: 1px solid #333; text-align: center; }
        .stat-number { font-size: 24px; font-weight: 700; color: #ffd700; }
        .stat-number.green { color: #00ff88; }
        .stat-number.red { color: #ff4444; }
        .stat-number.blue { color: #3498db; }
        .stat-label { font-size: 11px; color: #888; margin-top: 5px; text-transform: uppercase; }
        
        /* Filters */
        .filters-card { background: #1a1a1a; border-radius: 15px; padding: 20px; margin-bottom: 25px; border: 1px solid #333; }
        .filters-title { font-size: 14px; color: #ffd700; margin-bottom: 15px; font-weight: 600; }
        .filters-grid { display: grid; grid-template-columns: repeat(auto-fit, minmax(150px, 1fr)); gap: 15px; }
        .filter-group { }
        .filter-label { font-size: 11px; color: #888; margin-bottom: 5px; display: block; }
        .filter-select, .filter-input { width: 100%; padding: 12px; background: #252525; border: 1px solid #333; border-radius: 8px; color: #fff; font-size: 13px; outline: none; }
        .filter-select:focus, .filter-input:focus { border-color: #ffd700; }
        .filter-select option { background: #252525; }
        .filter-actions { display: flex; gap: 10px; align-items: flex-end; }
        .btn-filter { padding: 12px 25px; background: linear-gradient(135deg, #ffd700 0%, #ff8c00 100%); border: none; border-radius: 8px; color: #000; font-weight: 700; cursor: pointer; font-size: 13px; }
        .btn-reset { padding: 12px 20px; background: #333; border: none; border-radius: 8px; color: #888; font-weight: 600; cursor: pointer; font-size: 13px; text-decoration: none; }
        
        /* Bids Table */
        .bids-card { background: #1a1a1a; border-radius: 15px; border: 1px solid #333; overflow: hidden; }
        .bids-header { padding: 20px; border-bottom: 1px solid #333; display: flex; justify-content: space-between; align-items: center; flex-wrap: wrap; gap: 10px; }
        .bids-title { font-size: 18px; font-weight: 700; color: #ffd700; }
        .bids-count { font-size: 13px; color: #888; }
        
        .bids-table { width: 100%; overflow-x: auto; }
        .table-header { display: grid; grid-template-columns: 2fr 1.5fr 1fr 1fr 1fr 1fr 1.2fr; padding: 15px 20px; background: #252525; font-size: 11px; color: #888; text-transform: uppercase; min-width: 900px; }
        .table-row { display: grid; grid-template-columns: 2fr 1.5fr 1fr 1fr 1fr 1fr 1.2fr; padding: 15px 20px; border-bottom: 1px solid #333; align-items: center; min-width: 900px; }
        .table-row:hover { background: #252525; }
        
        .user-cell { display: flex; align-items: center; gap: 10px; }
        .user-avatar { width: 38px; height: 38px; border-radius: 50%; background: linear-gradient(135deg, #8b0000 0%, #5c0000 100%); display: flex; align-items: center; justify-content: center; font-weight: 700; font-size: 14px; color: #ffd700; flex-shrink: 0; }
        .user-info { min-width: 0; }
        .user-info h4 { font-size: 13px; margin-bottom: 2px; white-space: nowrap; overflow: hidden; text-overflow: ellipsis; }
        .user-info p { font-size: 11px; color: #888; }
        
        .game-cell { }
        .game-name { font-size: 13px; font-weight: 600; margin-bottom: 2px; }
        .game-name.deleted { color: #ff9800; font-style: italic; }
        .game-date { font-size: 11px; color: #888; }
        
        .type-badge { padding: 5px 12px; border-radius: 15px; font-size: 10px; font-weight: 700; text-transform: uppercase; display: inline-block; }
        .type-badge.jodi { background: rgba(52,152,219,0.2); color: #3498db; }
        .type-badge.andar { background: rgba(0,255,136,0.2); color: #00ff88; }
        .type-badge.bahar { background: rgba(255,68,68,0.2); color: #ff4444; }
        
        .number-cell { font-size: 22px; font-weight: 800; color: #ffd700; }
        
        .amount-cell { }
        .bid-amount { font-size: 14px; font-weight: 700; }
        .win-amount { font-size: 12px; color: #00ff88; margin-top: 2px; }
        
        .status-badge { padding: 6px 12px; border-radius: 15px; font-size: 10px; font-weight: 700; text-transform: uppercase; }
        .status-badge.pending { background: rgba(255,215,0,0.2); color: #ffd700; }
        .status-badge.won { background: rgba(0,255,136,0.2); color: #00ff88; }
        .status-badge.lost { background: rgba(255,68,68,0.2); color: #ff4444; }
        
        .date-cell { font-size: 12px; color: #888; }
        .date-cell .time { color: #555; font-size: 11px; }
        
        .empty-state { text-align: center; padding: 60px 20px; color: #555; }
        .empty-state .icon { font-size: 50px; margin-bottom: 15px; }
        
        /* Mobile Nav */
        .mobile-nav { display: none; position: fixed; bottom: 0; left: 0; right: 0; background: #1a1a1a; padding: 10px 0; border-top: 1px solid #333; z-index: 100; }
        .mobile-nav-inner { display: flex; justify-content: space-around; }
        .mobile-nav-item { text-align: center; color: #666; text-decoration: none; padding: 5px 10px; }
        .mobile-nav-item.active { color: #ffd700; }
        .mobile-nav-icon { font-size: 20px; display: block; margin-bottom: 3px; }
        .mobile-nav-text { font-size: 10px; }
        
        /* Scrollable table wrapper */
        .table-wrapper { overflow-x: auto; }
        
        @media (max-width: 900px) {
            .sidebar { display: none; }
            .main-content { margin-left: 0; padding-bottom: 80px; }
            .mobile-nav { display: block; }
        }
        
        @media (max-width: 600px) {
            .stats-row { grid-template-columns: repeat(2, 1fr); }
            .filters-grid { grid-template-columns: 1fr 1fr; }
        }
    </style>
</head>
<body>
    <div class="admin-layout">
        <aside class="sidebar">
            <div class="sidebar-header">
                <div class="sidebar-logo">🎰 SATTA KING</div>
                <div class="sidebar-subtitle">Admin Panel</div>
            </div>
            <nav class="sidebar-nav">
                <a href="index.php" class="nav-item"><span class="nav-icon">📊</span><span class="nav-text">Dashboard</span></a>
                <a href="users.php" class="nav-item"><span class="nav-icon">👥</span><span class="nav-text">Users</span><?php if ($pendingUsers > 0): ?><span class="nav-badge"><?= $pendingUsers ?></span><?php endif; ?></a>
                <a href="games.php" class="nav-item"><span class="nav-icon">🎮</span><span class="nav-text">Games</span></a>
                <a href="bids.php" class="nav-item active"><span class="nav-icon">🎯</span><span class="nav-text">All Bids</span></a>
                <a href="results.php" class="nav-item"><span class="nav-icon">🏆</span><span class="nav-text">Results</span></a>
                <a href="wallet.php" class="nav-item"><span class="nav-icon">💰</span><span class="nav-text">Wallet</span><?php if ($pendingRequests > 0): ?><span class="nav-badge"><?= $pendingRequests ?></span><?php endif; ?></a>
                <a href="settings.php" class="nav-item"><span class="nav-icon">⚙️</span><span class="nav-text">Settings</span></a>
                <a href="../logout.php" class="nav-item"><span class="nav-icon">🚪</span><span class="nav-text">Logout</span></a>
            </nav>
        </aside>

        <main class="main-content">
            <div class="top-bar">
                <h1 class="page-title">🎯 <span>All Bids</span></h1>
            </div>

            <!-- Stats -->
            <div class="stats-row">
                <div class="stat-box">
                    <div class="stat-number"><?= number_format($totalBids) ?></div>
                    <div class="stat-label">Total Bids</div>
                </div>
                <div class="stat-box">
                    <div class="stat-number blue"><?= number_format($todayBids) ?></div>
                    <div class="stat-label">Today's Bids</div>
                </div>
                <div class="stat-box">
                    <div class="stat-number">₹<?= number_format($totalBidAmount, 0) ?></div>
                    <div class="stat-label">Total Bid Amount</div>
                </div>
                <div class="stat-box">
                    <div class="stat-number red">₹<?= number_format($totalPayouts, 0) ?></div>
                    <div class="stat-label">Total Payouts</div>
                </div>
                <div class="stat-box">
                    <div class="stat-number <?= $profit >= 0 ? 'green' : 'red' ?>">₹<?= number_format($profit, 0) ?></div>
                    <div class="stat-label">Profit/Loss</div>
                </div>
            </div>

            <!-- Filters -->
            <div class="filters-card">
                <div class="filters-title">🔍 Filter Bids</div>
                <form method="GET">
                    <div class="filters-grid">
                        <div class="filter-group">
                            <label class="filter-label">Game</label>
                            <select name="game" class="filter-select">
                                <option value="">All Games</option>
                                <optgroup label="Active Games">
                                    <?php foreach ($games as $g): ?>
                                        <option value="<?= $g['id'] ?>" <?= $filter_game == $g['id'] ? 'selected' : '' ?>><?= htmlspecialchars($g['name']) ?></option>
                                    <?php endforeach; ?>
                                </optgroup>
                                <?php if (!empty($archivedGames)): ?>
                                    <optgroup label="Deleted Games">
                                        <?php foreach ($archivedGames as $g): ?>
                                            <option value="<?= $g['id'] ?>" <?= $filter_game == $g['id'] ? 'selected' : '' ?>><?= htmlspecialchars($g['name']) ?> (Deleted)</option>
                                        <?php endforeach; ?>
                                    </optgroup>
                                <?php endif; ?>
                            </select>
                        </div>
                        <div class="filter-group">
                            <label class="filter-label">Date</label>
                            <input type="date" name="date" class="filter-input" value="<?= $filter_date ?>">
                        </div>
                        <div class="filter-group">
                            <label class="filter-label">Status</label>
                            <select name="status" class="filter-select">
                                <option value="">All Status</option>
                                <option value="pending" <?= $filter_status == 'pending' ? 'selected' : '' ?>>Pending</option>
                                <option value="won" <?= $filter_status == 'won' ? 'selected' : '' ?>>Won</option>
                                <option value="lost" <?= $filter_status == 'lost' ? 'selected' : '' ?>>Lost</option>
                            </select>
                        </div>
                        <div class="filter-group">
                            <label class="filter-label">Bid Type</label>
                            <select name="type" class="filter-select">
                                <option value="">All Types</option>
                                <option value="jodi" <?= $filter_type == 'jodi' ? 'selected' : '' ?>>Jodi</option>
                                <option value="andar" <?= $filter_type == 'andar' ? 'selected' : '' ?>>Andar</option>
                                <option value="bahar" <?= $filter_type == 'bahar' ? 'selected' : '' ?>>Bahar</option>
                            </select>
                        </div>
                        <div class="filter-group">
                            <label class="filter-label">Search</label>
                            <input type="text" name="search" class="filter-input" placeholder="Name, Mobile, Number" value="<?= htmlspecialchars($search) ?>">
                        </div>
                        <div class="filter-actions">
                            <button type="submit" class="btn-filter">Apply</button>
                            <a href="bids.php" class="btn-reset">Reset</a>
                        </div>
                    </div>
                </form>
            </div>

            <!-- Bids Table -->
            <div class="bids-card">
                <div class="bids-header">
                    <div class="bids-title">📋 Bids List</div>
                    <div class="bids-count">Showing <?= count($bids) ?> bids</div>
                </div>
                
                <div class="table-wrapper">
                    <?php if (empty($bids)): ?>
                        <div class="empty-state">
                            <div class="icon">🎯</div>
                            <p>No bids found</p>
                        </div>
                    <?php else: ?>
                        <div class="table-header">
                            <div>User</div>
                            <div>Game</div>
                            <div>Type</div>
                            <div>Number</div>
                            <div>Amount</div>
                            <div>Status</div>
                            <div>Date</div>
                        </div>
                        <?php foreach ($bids as $bid): 
                            $isDeletedGame = empty($bid['current_rate']);
                        ?>
                            <div class="table-row">
                                <div class="user-cell">
                                    <div class="user-avatar"><?= strtoupper(substr($bid['user_name'], 0, 1)) ?></div>
                                    <div class="user-info">
                                        <h4><?= htmlspecialchars($bid['user_name']) ?></h4>
                                        <p><?= htmlspecialchars($bid['mobile']) ?></p>
                                    </div>
                                </div>
                                <div class="game-cell">
                                    <div class="game-name <?= $isDeletedGame ? 'deleted' : '' ?>">
                                        <?= htmlspecialchars($bid['game_name']) ?>
                                        <?php if ($isDeletedGame): ?>
                                            <span title="This game has been deleted">⚠️</span>
                                        <?php endif; ?>
                                    </div>
                                    <div class="game-date"><?= date('d M Y', strtotime($bid['playing_date'])) ?></div>
                                </div>
                                <div>
                                    <span class="type-badge <?= $bid['bid_type'] ?>"><?= $bid['bid_type'] ?></span>
                                </div>
                                <div class="number-cell"><?= $bid['number'] ?></div>
                                <div class="amount-cell">
                                    <div class="bid-amount">₹<?= number_format($bid['amount'], 0) ?></div>
                                    <?php if ($bid['status'] == 'won' && $bid['win_amount'] > 0): ?>
                                        <div class="win-amount">+₹<?= number_format($bid['win_amount'], 0) ?></div>
                                    <?php endif; ?>
                                </div>
                                <div>
                                    <span class="status-badge <?= $bid['status'] ?>"><?= $bid['status'] ?></span>
                                </div>
                                <div class="date-cell">
                                    <?= date('d M Y', strtotime($bid['created_at'])) ?>
                                    <div class="time"><?= date('h:i A', strtotime($bid['created_at'])) ?></div>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    <?php endif; ?>
                </div>
            </div>
        </main>
    </div>

    <!-- Mobile Nav -->
    <nav class="mobile-nav">
        <div class="mobile-nav-inner">
            <a href="index.php" class="mobile-nav-item"><span class="mobile-nav-icon">📊</span><span class="mobile-nav-text">Home</span></a>
            <a href="users.php" class="mobile-nav-item"><span class="mobile-nav-icon">👥</span><span class="mobile-nav-text">Users</span></a>
            <a href="bids.php" class="mobile-nav-item active"><span class="mobile-nav-icon">🎯</span><span class="mobile-nav-text">Bids</span></a>
            <a href="wallet.php" class="mobile-nav-item"><span class="mobile-nav-icon">💰</span><span class="mobile-nav-text">Wallet</span></a>
            <a href="settings.php" class="mobile-nav-item"><span class="mobile-nav-icon">⚙️</span><span class="mobile-nav-text">Settings</span></a>
        </div>
    </nav>
</body>
</html>