<?php
require_once 'config/db.php';
require_once 'config/auth.php';
require_once 'config/functions.php';
require_once 'config/security.php';

// Initialize secure session and check timeout
initSecureSession();
requireLogin();

// Check session timeout
if (!checkSessionTimeout($pdo)) {
    header("Location: login.php?msg=Session+expired");
    exit();
}

$user = getUser($pdo);
if (!$user) {
    session_destroy();
    header("Location: login.php");
    exit();
}

// Track session
trackUserSession($pdo, $user['id']);

$balance = getWalletBalance($pdo, $user['id']);

$error = '';
$success = '';
$showPinSetup = false;

// Check if transaction PIN is required and set
$requirePin = getSetting($pdo, 'require_transaction_pin', '1') == '1';
$hasPin = hasTransactionPin($pdo, $user['id']);

// Handle Transaction PIN Setup
if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['set_pin'])) {
    if (!verifyCSRFToken($_POST['csrf_token'] ?? '')) {
        $error = 'Invalid request. Please refresh.';
    } else {
        $pin = $_POST['new_pin'] ?? '';
        $confirmPin = $_POST['confirm_pin'] ?? '';
        
        if ($pin !== $confirmPin) {
            $error = 'PINs do not match!';
        } else {
            $result = setTransactionPin($pdo, $user['id'], $pin);
            if ($result['success']) {
                $success = 'Transaction PIN set successfully!';
                $hasPin = true;
            } else {
                $error = $result['message'];
            }
        }
    }
}

// Check pending request
$stmt = $pdo->prepare("SELECT * FROM wallet_requests WHERE user_id = ? AND status = 'pending'");
$stmt->execute([$user['id']]);
$pendingRequest = $stmt->fetch();

// Handle Deposit Request
if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['deposit'])) {
    if (!verifyCSRFToken($_POST['csrf_token'] ?? '')) {
        $error = 'Invalid request. Please refresh.';
    } else {
        $amount = floatval($_POST['amount']);
        $captcha = $_POST['captcha_answer'] ?? '';
        
        if ($captcha != $_SESSION['captcha_answer']) {
            $error = 'Wrong captcha!';
        } elseif ($pendingRequest) {
            $error = 'You already have a pending request!';
        } elseif ($amount < 100) {
            $error = 'Minimum deposit is ₹100';
        } elseif (!isset($_FILES['screenshot']) || $_FILES['screenshot']['error'] != 0) {
            $error = 'Please upload payment screenshot';
        } else {
            $file = $_FILES['screenshot'];
            $ext = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
            $allowed = ['jpg', 'jpeg', 'png', 'gif'];
            
            if (!in_array($ext, $allowed)) {
                $error = 'Only JPG, PNG, GIF allowed';
            } elseif ($file['size'] > 5 * 1024 * 1024) {
                $error = 'File too large (max 5MB)';
            } else {
                $filename = 'deposit_' . $user['id'] . '_' . time() . '.' . $ext;
                move_uploaded_file($file['tmp_name'], 'uploads/' . $filename);
                
                // Insert with archive columns
                $stmt = $pdo->prepare("INSERT INTO wallet_requests (user_id, user_name_archive, user_mobile_archive, type, amount, screenshot) VALUES (?, ?, ?, 'deposit', ?, ?)");
                $stmt->execute([$user['id'], $user['name'], $user['mobile'], $amount, $filename]);
                
                // Check and process referral bonus for first deposit
                processReferralBonus($pdo, $user['id']);
                
                $success = 'Deposit request submitted! We will verify and add money soon.';
                $pendingRequest = true;
                
                // Add notification
                addNotification($pdo, $user['id'], 'deposit_request', 'Deposit Request Submitted', 
                    "Your deposit request of ₹" . number_format($amount, 0) . " has been submitted.");
            }
        }
    }
}

// Handle Withdraw Request
if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['withdraw'])) {
    if (!verifyCSRFToken($_POST['csrf_token'] ?? '')) {
        $error = 'Invalid request. Please refresh.';
    } else {
        $amount = floatval($_POST['amount']);
        $upi_id = trim($_POST['upi_id']);
        $captcha = $_POST['captcha_answer'] ?? '';
        $pin = $_POST['transaction_pin'] ?? '';
        
        if ($captcha != $_SESSION['captcha_answer']) {
            $error = 'Wrong captcha!';
        } elseif ($requirePin && !$hasPin) {
            $error = 'Please set your Transaction PIN first!';
            $showPinSetup = true;
        } elseif ($requirePin && $hasPin) {
            // Verify PIN
            $pinCheck = verifyTransactionPin($pdo, $user['id'], $pin);
            if (!$pinCheck['valid']) {
                $error = 'Invalid Transaction PIN!';
                logSecurityEvent($user['id'], 'INVALID_PIN', 'medium', 'Invalid transaction PIN entered for withdraw');
            }
        }
        
        if (!$error) {
            if ($pendingRequest) {
                $error = 'You already have a pending request!';
            } elseif ($amount < 100) {
                $error = 'Minimum withdraw is ₹100';
            } elseif ($amount > $balance) {
                $error = 'Insufficient balance!';
            } elseif (empty($upi_id)) {
                $error = 'Enter your UPI ID';
            } else {
                // Check daily limit
                $limitCheck = checkDailyWithdrawLimit($pdo, $user['id'], $amount);
                if (!$limitCheck['allowed']) {
                    $error = $limitCheck['message'];
                } else {
                    $pdo->beginTransaction();
                    try {
                        // Insert withdraw request
                        $stmt = $pdo->prepare("INSERT INTO wallet_requests (user_id, user_name_archive, user_mobile_archive, type, amount, upi_id) VALUES (?, ?, ?, 'withdraw', ?, ?)");
                        $stmt->execute([$user['id'], $user['name'], $user['mobile'], $amount, $upi_id]);
                        $request_id = $pdo->lastInsertId();
                        
                        // IMMEDIATELY DEDUCT from wallet (hold the amount)
                        $stmt = $pdo->prepare("INSERT INTO wallet (user_id, user_name_archive, user_mobile_archive, type, amount, description, reference_type, reference_id) VALUES (?, ?, ?, 'debit', ?, ?, 'withdraw_hold', ?)");
                        $stmt->execute([$user['id'], $user['name'], $user['mobile'], $amount, 'Withdraw Request (Processing)', $request_id]);
                        
                        // Update daily withdraw total
                        updateDailyWithdrawTotal($pdo, $user['id'], $amount);
                        
                        // Add notification
                        addNotification($pdo, $user['id'], 'withdraw_request', 'Withdraw Request Submitted', 
                            "Your withdrawal request of ₹" . number_format($amount, 0) . " has been submitted.");
                        
                        $pdo->commit();
                        
                        $success = 'Withdraw request submitted! Amount held. Money will be sent to your UPI soon.';
                        $pendingRequest = true;
                        $balance -= $amount; // Update displayed balance
                        
                    } catch (Exception $e) {
                        $pdo->rollBack();
                        $error = 'Error submitting request. Please try again.';
                    }
                }
            }
        }
    }
}

// Get transaction history with archive support
$stmt = $pdo->prepare("SELECT * FROM wallet WHERE user_id = ? ORDER BY created_at DESC LIMIT 50");
$stmt->execute([$user['id']]);
$transactions = $stmt->fetchAll();

// Get wallet requests
$stmt = $pdo->prepare("SELECT * FROM wallet_requests WHERE user_id = ? ORDER BY created_at DESC LIMIT 20");
$stmt->execute([$user['id']]);
$requests = $stmt->fetchAll();

// Captcha
$num1 = rand(1, 9);
$num2 = rand(1, 9);
$_SESSION['captcha_answer'] = $num1 + $num2;

// CSRF Token
$csrfToken = generateCSRFToken();

// Get daily limit info
$dailyLimit = getSetting($pdo, 'daily_withdraw_limit', 50000);
$limitStatus = getRateLimitStatus($pdo, $user['id'], 'withdraw', 1000, 86400);

// Admin payment details
$admin_upi = getSetting($pdo, 'admin_upi', 'admin@upi');
$admin_qr = getSetting($pdo, 'admin_qr', '');
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Wallet - Satta King</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
            background: #0f0f0f;
            color: #fff;
            min-height: 100vh;
            padding-bottom: 80px;
        }
        
        /* Header */
        .header {
            background: linear-gradient(135deg, #8b0000 0%, #1a1a1a 100%);
            padding: 15px 20px;
            display: flex;
            justify-content: space-between;
            align-items: center;
            position: sticky;
            top: 0;
            z-index: 100;
        }
        
        .back-btn {
            color: #fff;
            text-decoration: none;
            font-size: 16px;
        }
        
        .page-title {
            font-size: 18px;
            font-weight: 700;
            color: #ffd700;
        }
        
        .header-space {
            width: 50px;
        }
        
        /* Container */
        .container {
            padding: 20px;
            max-width: 500px;
            margin: 0 auto;
        }
        
        /* Balance Card */
        .balance-card {
            background: linear-gradient(135deg, #1a1a2e 0%, #16213e 100%);
            border-radius: 20px;
            padding: 30px;
            text-align: center;
            margin-bottom: 25px;
            border: 1px solid #333;
            position: relative;
            overflow: hidden;
        }
        
        .balance-card::before {
            content: '💰';
            position: absolute;
            font-size: 100px;
            opacity: 0.1;
            right: -20px;
            top: -20px;
        }
        
        .balance-label {
            font-size: 14px;
            color: #888;
            margin-bottom: 5px;
        }
        
        .balance-amount {
            font-size: 42px;
            font-weight: 800;
            color: #00ff88;
        }
        
        .balance-actions {
            display: flex;
            gap: 15px;
            margin-top: 20px;
        }
        
        .balance-btn {
            flex: 1;
            padding: 12px;
            border-radius: 25px;
            font-size: 14px;
            font-weight: 700;
            cursor: pointer;
            border: none;
            transition: all 0.3s;
        }
        
        .balance-btn.deposit {
            background: linear-gradient(135deg, #00c853 0%, #00a844 100%);
            color: #fff;
        }
        
        .balance-btn.withdraw {
            background: linear-gradient(135deg, #ff5722 0%, #e64a19 100%);
            color: #fff;
        }
        
        /* Alert */
        .alert {
            padding: 15px;
            border-radius: 10px;
            margin-bottom: 20px;
            font-size: 14px;
        }
        
        .alert-error {
            background: rgba(255,0,0,0.2);
            border: 1px solid #ff4444;
            color: #ff6666;
        }
        
        .alert-success {
            background: rgba(0,255,0,0.2);
            border: 1px solid #00cc00;
            color: #00ff00;
        }
        
        .alert-warning {
            background: rgba(255,215,0,0.2);
            border: 1px solid #ffd700;
            color: #ffd700;
        }
        
        /* Pending Request Card */
        .pending-card {
            background: linear-gradient(135deg, #332701 0%, #1a1400 100%);
            border: 1px solid #ffd700;
            border-radius: 15px;
            padding: 20px;
            margin-bottom: 25px;
            text-align: center;
        }
        
        .pending-icon {
            font-size: 40px;
            margin-bottom: 10px;
        }
        
        .pending-text {
            color: #ffd700;
            font-size: 14px;
        }
        
        /* Tabs */
        .tabs {
            display: flex;
            background: #1a1a1a;
            border-radius: 12px;
            padding: 5px;
            margin-bottom: 20px;
        }
        
        .tab {
            flex: 1;
            padding: 12px;
            text-align: center;
            border-radius: 10px;
            font-size: 13px;
            font-weight: 600;
            color: #888;
            cursor: pointer;
            transition: all 0.3s;
        }
        
        .tab.active {
            background: linear-gradient(135deg, #8b0000 0%, #5c0000 100%);
            color: #fff;
        }
        
        /* Tab Content */
        .tab-content {
            display: none;
        }
        
        .tab-content.active {
            display: block;
        }
        
        /* Form Section */
        .form-section {
            background: #1a1a1a;
            border-radius: 15px;
            padding: 20px;
            margin-bottom: 20px;
            border: 1px solid #333;
        }
        
        .section-title {
            color: #ffd700;
            font-size: 16px;
            margin-bottom: 20px;
            display: flex;
            align-items: center;
            gap: 10px;
        }
        
        /* Payment Info */
        .payment-info {
            background: #252525;
            border-radius: 10px;
            padding: 15px;
            margin-bottom: 20px;
            text-align: center;
        }
        
        .payment-label {
            font-size: 12px;
            color: #888;
            margin-bottom: 5px;
        }
        
        .payment-upi {
            font-size: 18px;
            font-weight: 700;
            color: #ffd700;
            margin-bottom: 10px;
        }
        
        .copy-btn {
            background: #333;
            border: none;
            color: #fff;
            padding: 8px 20px;
            border-radius: 20px;
            font-size: 12px;
            cursor: pointer;
        }
        
        .qr-image {
            max-width: 200px;
            margin: 15px auto;
            display: block;
            border-radius: 10px;
        }
        
        /* Form */
        .form-group {
            margin-bottom: 20px;
        }
        
        .form-label {
            display: block;
            font-size: 13px;
            color: #888;
            margin-bottom: 8px;
        }
        
        .form-input {
            width: 100%;
            padding: 15px;
            background: #252525;
            border: 2px solid #333;
            border-radius: 10px;
            color: #fff;
            font-size: 16px;
            outline: none;
        }
        
        .form-input:focus {
            border-color: #ffd700;
        }
        
        .form-input::placeholder {
            color: #555;
        }
        
        .file-input {
            display: none;
        }
        
        .file-label {
            display: block;
            padding: 20px;
            background: #252525;
            border: 2px dashed #444;
            border-radius: 10px;
            text-align: center;
            cursor: pointer;
            transition: all 0.3s;
        }
        
        .file-label:hover {
            border-color: #ffd700;
        }
        
        .file-icon {
            font-size: 30px;
            margin-bottom: 10px;
        }
        
        .file-text {
            font-size: 13px;
            color: #888;
        }
        
        .file-name {
            color: #00ff88;
            font-size: 13px;
            margin-top: 10px;
        }
        
        /* Captcha */
        .captcha-row {
            display: flex;
            align-items: center;
            gap: 15px;
            margin-bottom: 20px;
        }
        
        .captcha-question {
            font-size: 20px;
            color: #ffd700;
            font-weight: 700;
        }
        
        .captcha-input {
            width: 80px;
            padding: 12px;
            background: #252525;
            border: 2px solid #333;
            border-radius: 10px;
            color: #fff;
            font-size: 18px;
            text-align: center;
            outline: none;
        }
        
        .captcha-input:focus {
            border-color: #ffd700;
        }
        
        /* Submit Button */
        .btn-submit {
            width: 100%;
            padding: 16px;
            border: none;
            border-radius: 10px;
            font-size: 16px;
            font-weight: 700;
            cursor: pointer;
            text-transform: uppercase;
        }
        
        .btn-submit.deposit {
            background: linear-gradient(135deg, #00c853 0%, #00a844 100%);
            color: #fff;
        }
        
        .btn-submit.withdraw {
            background: linear-gradient(135deg, #ff5722 0%, #e64a19 100%);
            color: #fff;
        }
        
        /* Transaction List */
        .txn-list {
            margin-top: 25px;
        }
        
        .txn-title {
            color: #ffd700;
            font-size: 16px;
            margin-bottom: 15px;
            padding-bottom: 10px;
            border-bottom: 1px solid #333;
        }
        
        .txn-item {
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding: 15px;
            background: #1a1a1a;
            border-radius: 12px;
            margin-bottom: 10px;
            border: 1px solid #333;
        }
        
        .txn-left {
            display: flex;
            align-items: center;
            gap: 12px;
        }
        
        .txn-icon {
            width: 45px;
            height: 45px;
            border-radius: 12px;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 20px;
        }
        
        .txn-icon.credit {
            background: rgba(0,255,136,0.2);
        }
        
        .txn-icon.debit {
            background: rgba(255,68,68,0.2);
        }
        
        .txn-info h4 {
            font-size: 14px;
            font-weight: 600;
            margin-bottom: 3px;
        }
        
        .txn-info p {
            font-size: 11px;
            color: #666;
        }
        
        .txn-amount {
            font-size: 16px;
            font-weight: 700;
        }
        
        .txn-amount.credit {
            color: #00ff88;
        }
        
        .txn-amount.debit {
            color: #ff4444;
        }
        
        /* Request Status */
        .request-item {
            padding: 15px;
            background: #1a1a1a;
            border-radius: 12px;
            margin-bottom: 10px;
            border: 1px solid #333;
        }
        
        .request-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 10px;
        }
        
        .request-type {
            font-size: 12px;
            font-weight: 700;
            padding: 4px 10px;
            border-radius: 15px;
        }
        
        .request-type.deposit {
            background: rgba(0,200,83,0.2);
            color: #00c853;
        }
        
        .request-type.withdraw {
            background: rgba(255,87,34,0.2);
            color: #ff5722;
        }
        
        .request-status {
            font-size: 11px;
            font-weight: 700;
            padding: 4px 10px;
            border-radius: 15px;
        }
        
        .request-status.pending {
            background: rgba(255,215,0,0.2);
            color: #ffd700;
        }
        
        .request-status.approved {
            background: rgba(0,255,136,0.2);
            color: #00ff88;
        }
        
        .request-status.rejected {
            background: rgba(255,68,68,0.2);
            color: #ff4444;
        }
        
        .request-amount {
            font-size: 20px;
            font-weight: 700;
            color: #ffd700;
        }
        
        .request-date {
            font-size: 11px;
            color: #666;
            margin-top: 5px;
        }
        
        /* Empty State */
        .empty-state {
            text-align: center;
            padding: 40px 20px;
            color: #555;
        }
        
        /* Bottom Navigation */
        .bottom-nav {
            position: fixed;
            bottom: 0;
            left: 0;
            right: 0;
            background: #1a1a1a;
            display: flex;
            justify-content: space-around;
            padding: 10px 0;
            border-top: 1px solid #333;
            z-index: 100;
        }
        
        .nav-item {
            text-align: center;
            color: #666;
            text-decoration: none;
            padding: 5px 15px;
        }
        
        .nav-item.active {
            color: #ffd700;
        }
        
        .nav-icon {
            font-size: 22px;
            display: block;
            margin-bottom: 4px;
        }
        
        .nav-label {
            font-size: 10px;
        }
    </style>
</head>
<body>
    <header class="header">
        <a href="index.php" class="back-btn">← Back</a>
        <div class="page-title">💰 Wallet</div>
        <div class="header-space"></div>
    </header>

    <div class="container">
        <?php if ($error): ?>
            <div class="alert alert-error"><?= $error ?></div>
        <?php endif; ?>
        <?php if ($success): ?>
            <div class="alert alert-success"><?= $success ?></div>
        <?php endif; ?>

        <!-- Balance Card -->
        <div class="balance-card">
            <div class="balance-label">Available Balance</div>
            <div class="balance-amount">₹<?= number_format($balance, 2) ?></div>
            <div class="balance-actions">
                <button class="balance-btn deposit" onclick="switchTab('deposit')">+ Deposit</button>
                <button class="balance-btn withdraw" onclick="switchTab('withdraw')">- Withdraw</button>
            </div>
        </div>

        <?php if ($pendingRequest && is_array($pendingRequest)): ?>
            <div class="pending-card">
                <div class="pending-icon">⏳</div>
                <div class="pending-text">
                    You have a pending <?= $pendingRequest['type'] ?> request of ₹<?= number_format($pendingRequest['amount'], 0) ?>
                </div>
            </div>
        <?php endif; ?>

        <!-- Tabs -->
        <div class="tabs">
            <div class="tab active" onclick="switchTab('deposit')">💵 Deposit</div>
            <div class="tab" onclick="switchTab('withdraw')">💸 Withdraw</div>
            <div class="tab" onclick="switchTab('history')">📋 History</div>
        </div>

        <!-- Deposit Tab -->
        <div class="tab-content active" id="depositTab">
            <div class="form-section">
                <div class="section-title">💵 Add Money</div>
                
                <div class="payment-info">
                    <div class="payment-label">Pay to this UPI ID</div>
                    <div class="payment-upi" id="upiId"><?= htmlspecialchars($admin_upi) ?></div>
                    <button class="copy-btn" onclick="copyUPI()">📋 Copy UPI</button>
                    <?php if ($admin_qr && file_exists('uploads/' . $admin_qr)): ?>
                        <img src="uploads/<?= $admin_qr ?>" class="qr-image" alt="QR Code">
                    <?php endif; ?>
                </div>
                
                <?php if (!$pendingRequest): ?>
                    <form method="POST" enctype="multipart/form-data">
                        <input type="hidden" name="csrf_token" value="<?= $csrfToken ?>">
                        <div class="form-group">
                            <label class="form-label">Amount (₹)</label>
                            <input type="number" name="amount" class="form-input" placeholder="Enter amount" min="100" required>
                        </div>
                        
                        <div class="form-group">
                            <label class="form-label">Payment Screenshot</label>
                            <input type="file" name="screenshot" id="screenshot" class="file-input" accept="image/*" required>
                            <label for="screenshot" class="file-label">
                                <div class="file-icon">📷</div>
                                <div class="file-text">Click to upload payment screenshot</div>
                                <div class="file-name" id="fileName"></div>
                            </label>
                        </div>
                        
                        <div class="captcha-row">
                            <span class="captcha-question"><?= $num1 ?> + <?= $num2 ?> = ?</span>
                            <input type="number" name="captcha_answer" class="captcha-input" required>
                        </div>
                        
                        <button type="submit" name="deposit" class="btn-submit deposit">Submit Deposit Request</button>
                    </form>
                <?php else: ?>
                    <div class="alert alert-warning">You have a pending request. Please wait for it to be processed.</div>
                <?php endif; ?>
            </div>
        </div>

        <!-- Withdraw Tab -->
        <div class="tab-content" id="withdrawTab">
            <div class="form-section">
                <div class="section-title">💸 Withdraw Money</div>
                
                <!-- PIN Setup Section -->
                <?php if ($requirePin && !$hasPin): ?>
                    <div class="alert alert-warning" style="margin-bottom:20px;">
                        ⚠️ Please set your Transaction PIN first to withdraw money.
                    </div>
                    <form method="POST" style="margin-bottom:30px;padding:20px;background:#1a1a1a;border-radius:12px;border:1px solid #ffd700;">
                        <input type="hidden" name="csrf_token" value="<?= $csrfToken ?>">
                        <div class="section-title" style="font-size:16px;color:#ffd700;">🔐 Set Transaction PIN</div>
                        <div class="form-group">
                            <label class="form-label">New 4-Digit PIN</label>
                            <input type="password" name="new_pin" class="form-input" placeholder="Enter 4 digit PIN" maxlength="4" pattern="[0-9]{4}" required>
                        </div>
                        <div class="form-group">
                            <label class="form-label">Confirm PIN</label>
                            <input type="password" name="confirm_pin" class="form-input" placeholder="Confirm PIN" maxlength="4" pattern="[0-9]{4}" required>
                        </div>
                        <button type="submit" name="set_pin" class="btn-submit" style="background:linear-gradient(135deg, #ffd700, #ff8c00);">Set PIN</button>
                    </form>
                <?php endif; ?>
                
                <!-- Daily Limit Info -->
                <div style="background:#1a1a2e;padding:12px;border-radius:8px;margin-bottom:15px;font-size:13px;">
                    <span style="color:#888;">Daily Limit:</span> 
                    <span style="color:#ffd700;">₹<?= number_format($dailyLimit, 0) ?></span>
                </div>
                
                <?php if (!$pendingRequest): ?>
                    <form method="POST">
                        <input type="hidden" name="csrf_token" value="<?= $csrfToken ?>">
                        <div class="form-group">
                            <label class="form-label">Amount (₹)</label>
                            <input type="number" name="amount" class="form-input" placeholder="Enter amount" min="100" max="<?= $balance ?>" required>
                            <p style="font-size: 12px; color: #888; margin-top: 8px;">Available: ₹<?= number_format($balance, 2) ?></p>
                        </div>
                        
                        <div class="form-group">
                            <label class="form-label">Your UPI ID</label>
                            <input type="text" name="upi_id" class="form-input" placeholder="yourname@upi" value="<?= htmlspecialchars($user['upi_id'] ?? '') ?>" required>
                        </div>
                        
                        <?php if ($requirePin && $hasPin): ?>
                        <div class="form-group">
                            <label class="form-label">🔐 Transaction PIN</label>
                            <input type="password" name="transaction_pin" class="form-input" placeholder="Enter 4 digit PIN" maxlength="4" pattern="[0-9]{4}" required>
                        </div>
                        <?php endif; ?>
                        
                        <div class="captcha-row">
                            <span class="captcha-question"><?= $num1 ?> + <?= $num2 ?> = ?</span>
                            <input type="number" name="captcha_answer" class="captcha-input" required>
                        </div>
                        
                        <button type="submit" name="withdraw" class="btn-submit withdraw" <?= ($requirePin && !$hasPin) ? 'disabled style="opacity:0.5;cursor:not-allowed;"' : '' ?>>Submit Withdraw Request</button>
                    </form>
                <?php else: ?>
                    <div class="alert alert-warning">You have a pending request. Please wait for it to be processed.</div>
                <?php endif; ?>
            </div>
        </div>

        <!-- History Tab -->
        <div class="tab-content" id="historyTab">
            <!-- Requests -->
            <div class="txn-title">📝 Requests</div>
            <?php if (empty($requests)): ?>
                <div class="empty-state">No requests yet</div>
            <?php else: ?>
                <?php foreach ($requests as $req): ?>
                    <div class="request-item">
                        <div class="request-header">
                            <span class="request-type <?= $req['type'] ?>"><?= strtoupper($req['type']) ?></span>
                            <span class="request-status <?= $req['status'] ?>"><?= strtoupper($req['status']) ?></span>
                        </div>
                        <div class="request-amount">₹<?= number_format($req['amount'], 0) ?></div>
                        <div class="request-date"><?= date('d M Y, h:i A', strtotime($req['created_at'])) ?></div>
                    </div>
                <?php endforeach; ?>
            <?php endif; ?>

            <!-- Transactions -->
            <div class="txn-list">
                <div class="txn-title">💳 Transactions</div>
                <?php if (empty($transactions)): ?>
                    <div class="empty-state">No transactions yet</div>
                <?php else: ?>
                    <?php foreach ($transactions as $txn): ?>
                        <div class="txn-item">
                            <div class="txn-left">
                                <div class="txn-icon <?= $txn['type'] ?>">
                                    <?= $txn['type'] == 'credit' ? '⬇️' : '⬆️' ?>
                                </div>
                                <div class="txn-info">
                                    <h4><?= htmlspecialchars($txn['description']) ?></h4>
                                    <p><?= date('d M Y, h:i A', strtotime($txn['created_at'])) ?></p>
                                </div>
                            </div>
                            <div class="txn-amount <?= $txn['type'] ?>">
                                <?= $txn['type'] == 'credit' ? '+' : '-' ?>₹<?= number_format($txn['amount'], 0) ?>
                            </div>
                        </div>
                    <?php endforeach; ?>
                <?php endif; ?>
            </div>
        </div>
    </div>

    <nav class="bottom-nav">
        <a href="index.php" class="nav-item">
            <span class="nav-icon">🏠</span>
            <span class="nav-label">Home</span>
        </a>
        <a href="wallet.php" class="nav-item active">
            <span class="nav-icon">💰</span>
            <span class="nav-label">Wallet</span>
        </a>
        <a href="history.php" class="nav-item">
            <span class="nav-icon">📋</span>
            <span class="nav-label">History</span>
        </a>
        <a href="profile.php" class="nav-item">
            <span class="nav-icon">👤</span>
            <span class="nav-label">Profile</span>
        </a>
    </nav>

    <script>
        function switchTab(tab) {
            // Update tabs
            document.querySelectorAll('.tab').forEach((t, i) => {
                t.classList.remove('active');
                if ((tab === 'deposit' && i === 0) || (tab === 'withdraw' && i === 1) || (tab === 'history' && i === 2)) {
                    t.classList.add('active');
                }
            });
            
            // Update content
            document.querySelectorAll('.tab-content').forEach(c => c.classList.remove('active'));
            document.getElementById(tab + 'Tab').classList.add('active');
        }
        
        function copyUPI() {
            const upi = document.getElementById('upiId').textContent;
            navigator.clipboard.writeText(upi).then(() => {
                alert('UPI ID copied!');
            });
        }
        
        // File input display
        document.getElementById('screenshot')?.addEventListener('change', function() {
            const fileName = this.files[0]?.name || '';
            document.getElementById('fileName').textContent = fileName;
        });
    </script>
</body>
</html>
