<?php
require_once 'config/db.php';
require_once 'config/auth.php';
require_once 'config/functions.php';
require_once 'config/security.php';

initSecureSession();
requireLogin();

if (!checkSessionTimeout($pdo)) {
    header("Location: login.php?msg=Session+expired");
    exit();
}

$user = getUser($pdo);
if (!$user) {
    session_destroy();
    header("Location: login.php");
    exit();
}

trackUserSession($pdo, $user['id']);

$balance = getWalletBalance($pdo, $user['id']);

$error = '';
$success = '';
$csrfToken = generateCSRFToken();

// Handle Profile Update
if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['update_profile'])) {
    if (!verifyCSRFToken($_POST['csrf_token'] ?? '')) {
        $error = 'Invalid request. Please refresh.';
    } else {
    $name = trim($_POST['name']);
    $bank_name = trim($_POST['bank_name'] ?? '');
    $account_no = trim($_POST['account_no'] ?? '');
    $account_holder = trim($_POST['account_holder'] ?? '');
    $ifsc_code = trim($_POST['ifsc_code'] ?? '');
    $upi_id = trim($_POST['upi_id'] ?? '');
    
    if (empty($name)) {
        $error = 'Name is required';
    } else {
        // Handle profile photo upload
        $photo = $user['profile_photo'];
        if (isset($_FILES['profile_photo']) && $_FILES['profile_photo']['error'] == 0) {
            $file = $_FILES['profile_photo'];
            $ext = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
            $allowed = ['jpg', 'jpeg', 'png', 'gif'];
            
            if (in_array($ext, $allowed) && $file['size'] <= 2 * 1024 * 1024) {
                $filename = 'profile_' . $user['id'] . '_' . time() . '.' . $ext;
                if (move_uploaded_file($file['tmp_name'], 'uploads/' . $filename)) {
                    // Delete old photo
                    if ($photo && file_exists('uploads/' . $photo)) {
                        unlink('uploads/' . $photo);
                    }
                    $photo = $filename;
                }
            }
        }
        
        // Handle QR code upload
        $qr = $user['qr_code'];
        if (isset($_FILES['qr_code']) && $_FILES['qr_code']['error'] == 0) {
            $file = $_FILES['qr_code'];
            $ext = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
            $allowed = ['jpg', 'jpeg', 'png', 'gif'];
            
            if (in_array($ext, $allowed) && $file['size'] <= 2 * 1024 * 1024) {
                $filename = 'qr_' . $user['id'] . '_' . time() . '.' . $ext;
                if (move_uploaded_file($file['tmp_name'], 'uploads/' . $filename)) {
                    if ($qr && file_exists('uploads/' . $qr)) {
                        unlink('uploads/' . $qr);
                    }
                    $qr = $filename;
                }
            }
        }
        
        $stmt = $pdo->prepare("UPDATE users SET name = ?, bank_name = ?, account_no = ?, account_holder = ?, ifsc_code = ?, upi_id = ?, profile_photo = ?, qr_code = ? WHERE id = ?");
        $stmt->execute([$name, $bank_name, $account_no, $account_holder, $ifsc_code, $upi_id, $photo, $qr, $user['id']]);
        
        $success = 'Profile updated successfully!';
        $user = getUser($pdo); // Refresh user data
    }
    }
}

// Handle Password Change
if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['change_password'])) {
    if (!verifyCSRFToken($_POST['csrf_token'] ?? '')) {
        $error = 'Invalid request. Please refresh.';
    } else {
    $current = $_POST['current_password'];
    $new = $_POST['new_password'];
    $confirm = $_POST['confirm_password'];
    
    if (!password_verify($current, $user['password'])) {
        $error = 'Current password is incorrect';
    } elseif (strlen($new) < 6) {
        $error = 'New password must be at least 6 characters';
    } elseif ($new !== $confirm) {
        $error = 'Passwords do not match';
    } else {
        $hash = password_hash($new, PASSWORD_DEFAULT);
        $stmt = $pdo->prepare("UPDATE users SET password = ? WHERE id = ?");
        $stmt->execute([$hash, $user['id']]);
        $success = 'Password changed successfully!';
    }
    }
}

// Get user stats
$stmt = $pdo->prepare("SELECT COUNT(*) as total, SUM(amount) as total_amount FROM bids WHERE user_id = ?");
$stmt->execute([$user['id']]);
$bidStats = $stmt->fetch();

$stmt = $pdo->prepare("SELECT COUNT(*) as wins, SUM(win_amount) as win_amount FROM bids WHERE user_id = ? AND status = 'won'");
$stmt->execute([$user['id']]);
$winStats = $stmt->fetch();

$avatar = $user['profile_photo'] ? 'uploads/' . $user['profile_photo'] : '';
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Profile - Satta King</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
            background: #0f0f0f;
            color: #fff;
            min-height: 100vh;
            padding-bottom: 80px;
        }
        
        /* Header */
        .header {
            background: linear-gradient(135deg, #8b0000 0%, #1a1a1a 100%);
            padding: 15px 20px;
            display: flex;
            justify-content: space-between;
            align-items: center;
            position: sticky;
            top: 0;
            z-index: 100;
        }
        
        .back-btn {
            color: #fff;
            text-decoration: none;
            font-size: 16px;
        }
        
        .page-title {
            font-size: 18px;
            font-weight: 700;
            color: #ffd700;
        }
        
        .logout-btn {
            color: #ff4444;
            text-decoration: none;
            font-size: 14px;
            font-weight: 600;
        }
        
        /* Container */
        .container {
            padding: 20px;
            max-width: 500px;
            margin: 0 auto;
        }
        
        /* Alert */
        .alert {
            padding: 15px;
            border-radius: 10px;
            margin-bottom: 20px;
            font-size: 14px;
        }
        
        .alert-error {
            background: rgba(255,0,0,0.2);
            border: 1px solid #ff4444;
            color: #ff6666;
        }
        
        .alert-success {
            background: rgba(0,255,0,0.2);
            border: 1px solid #00cc00;
            color: #00ff00;
        }
        
        /* Profile Header */
        .profile-header {
            text-align: center;
            margin-bottom: 25px;
        }
        
        .avatar-container {
            position: relative;
            width: 120px;
            height: 120px;
            margin: 0 auto 15px;
        }
        
        .avatar {
            width: 120px;
            height: 120px;
            border-radius: 50%;
            border: 4px solid #ffd700;
            object-fit: cover;
            background: #333;
        }
        
        .avatar-placeholder {
            width: 120px;
            height: 120px;
            border-radius: 50%;
            border: 4px solid #ffd700;
            background: linear-gradient(135deg, #8b0000 0%, #5c0000 100%);
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 48px;
            font-weight: 700;
            color: #ffd700;
        }
        
        .profile-name {
            font-size: 24px;
            font-weight: 700;
            margin-bottom: 5px;
        }
        
        .profile-mobile {
            color: #888;
            font-size: 14px;
            margin-bottom: 5px;
        }
        
        .profile-status {
            display: inline-block;
            padding: 5px 15px;
            border-radius: 20px;
            font-size: 12px;
            font-weight: 600;
        }
        
        .profile-status.active {
            background: rgba(0,255,136,0.2);
            color: #00ff88;
        }
        
        .profile-status.pending {
            background: rgba(255,215,0,0.2);
            color: #ffd700;
        }
        
        /* Stats Grid */
        .stats-grid {
            display: grid;
            grid-template-columns: repeat(3, 1fr);
            gap: 10px;
            margin-bottom: 25px;
        }
        
        .stat-card {
            background: linear-gradient(145deg, #1e1e1e 0%, #2a2a2a 100%);
            border-radius: 12px;
            padding: 15px 10px;
            text-align: center;
            border: 1px solid #333;
        }
        
        .stat-value {
            font-size: 20px;
            font-weight: 700;
            color: #ffd700;
        }
        
        .stat-value.green {
            color: #00ff88;
        }
        
        .stat-label {
            font-size: 10px;
            color: #888;
            margin-top: 5px;
        }
        
        /* Section Card */
        .section-card {
            background: #1a1a1a;
            border-radius: 15px;
            padding: 20px;
            margin-bottom: 20px;
            border: 1px solid #333;
        }
        
        .section-title {
            color: #ffd700;
            font-size: 16px;
            margin-bottom: 20px;
            display: flex;
            align-items: center;
            gap: 10px;
            padding-bottom: 10px;
            border-bottom: 1px solid #333;
        }
        
        /* Form */
        .form-group {
            margin-bottom: 18px;
        }
        
        .form-label {
            display: block;
            font-size: 13px;
            color: #888;
            margin-bottom: 8px;
        }
        
        .form-input {
            width: 100%;
            padding: 14px;
            background: #252525;
            border: 2px solid #333;
            border-radius: 10px;
            color: #fff;
            font-size: 15px;
            outline: none;
        }
        
        .form-input:focus {
            border-color: #ffd700;
        }
        
        .form-input:disabled {
            opacity: 0.6;
            cursor: not-allowed;
        }
        
        .form-input::placeholder {
            color: #555;
        }
        
        .form-row {
            display: grid;
            grid-template-columns: repeat(2, 1fr);
            gap: 15px;
        }
        
        /* File Upload */
        .file-upload {
            margin-bottom: 18px;
        }
        
        .file-label {
            display: block;
            font-size: 13px;
            color: #888;
            margin-bottom: 8px;
        }
        
        .file-input {
            display: none;
        }
        
        .file-box {
            display: flex;
            align-items: center;
            gap: 15px;
            padding: 15px;
            background: #252525;
            border: 2px dashed #444;
            border-radius: 10px;
            cursor: pointer;
            transition: all 0.3s;
        }
        
        .file-box:hover {
            border-color: #ffd700;
        }
        
        .file-preview {
            width: 60px;
            height: 60px;
            border-radius: 10px;
            object-fit: cover;
            background: #333;
        }
        
        .file-placeholder {
            width: 60px;
            height: 60px;
            border-radius: 10px;
            background: #333;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 24px;
        }
        
        .file-text {
            flex: 1;
        }
        
        .file-text h4 {
            font-size: 14px;
            margin-bottom: 3px;
        }
        
        .file-text p {
            font-size: 11px;
            color: #666;
        }
        
        /* Submit Button */
        .btn-submit {
            width: 100%;
            padding: 15px;
            background: linear-gradient(135deg, #00c853 0%, #00a844 100%);
            border: none;
            border-radius: 10px;
            color: #fff;
            font-size: 15px;
            font-weight: 700;
            cursor: pointer;
            text-transform: uppercase;
        }
        
        .btn-submit:hover {
            opacity: 0.9;
        }
        
        .btn-submit.red {
            background: linear-gradient(135deg, #ff5722 0%, #e64a19 100%);
        }
        
        /* Info Display */
        .info-row {
            display: flex;
            justify-content: space-between;
            padding: 12px 0;
            border-bottom: 1px solid #333;
        }
        
        .info-row:last-child {
            border-bottom: none;
        }
        
        .info-label {
            color: #888;
            font-size: 14px;
        }
        
        .info-value {
            font-size: 14px;
            font-weight: 600;
        }
        
        /* Quick Links */
        .quick-links {
            display: flex;
            flex-direction: column;
            gap: 10px;
        }
        
        .quick-link {
            display: flex;
            align-items: center;
            justify-content: space-between;
            padding: 15px;
            background: #252525;
            border-radius: 10px;
            text-decoration: none;
            color: #fff;
            transition: all 0.3s;
        }
        
        .quick-link:hover {
            background: #333;
        }
        
        .quick-link-left {
            display: flex;
            align-items: center;
            gap: 12px;
        }
        
        .quick-link-icon {
            width: 40px;
            height: 40px;
            border-radius: 10px;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 18px;
        }
        
        .quick-link-icon.green { background: rgba(0,200,83,0.2); }
        .quick-link-icon.blue { background: rgba(52,152,219,0.2); }
        .quick-link-icon.orange { background: rgba(255,152,0,0.2); }
        .quick-link-icon.red { background: rgba(255,68,68,0.2); }
        
        .quick-link-text h4 {
            font-size: 14px;
            margin-bottom: 2px;
        }
        
        .quick-link-text p {
            font-size: 11px;
            color: #666;
        }
        
        .quick-link-arrow {
            color: #666;
            font-size: 18px;
        }
        
        /* Bottom Navigation */
        .bottom-nav {
            position: fixed;
            bottom: 0;
            left: 0;
            right: 0;
            background: #1a1a1a;
            display: flex;
            justify-content: space-around;
            padding: 10px 0;
            border-top: 1px solid #333;
            z-index: 100;
        }
        
        .nav-item {
            text-align: center;
            color: #666;
            text-decoration: none;
            padding: 5px 15px;
        }
        
        .nav-item.active {
            color: #ffd700;
        }
        
        .nav-icon {
            font-size: 22px;
            display: block;
            margin-bottom: 4px;
        }
        
        .nav-label {
            font-size: 10px;
        }
        
        /* Responsive */
        @media (max-width: 400px) {
            .stats-grid {
                grid-template-columns: repeat(3, 1fr);
                gap: 8px;
            }
            .stat-card {
                padding: 12px 8px;
            }
            .stat-value {
                font-size: 18px;
            }
            .form-row {
                grid-template-columns: 1fr;
            }
        }
    </style>
</head>
<body>
    <header class="header">
        <a href="index.php" class="back-btn">← Back</a>
        <div class="page-title">👤 Profile</div>
        <a href="logout.php" class="logout-btn">Logout</a>
    </header>

    <div class="container">
        <?php if ($error): ?>
            <div class="alert alert-error"><?= $error ?></div>
        <?php endif; ?>
        <?php if ($success): ?>
            <div class="alert alert-success"><?= $success ?></div>
        <?php endif; ?>

        <!-- Profile Header -->
        <div class="profile-header">
            <div class="avatar-container">
                <?php if ($avatar && file_exists($avatar)): ?>
                    <img src="<?= $avatar ?>" class="avatar" alt="Profile">
                <?php else: ?>
                    <div class="avatar-placeholder"><?= strtoupper(substr($user['name'], 0, 1)) ?></div>
                <?php endif; ?>
            </div>
            <div class="profile-name"><?= htmlspecialchars($user['name']) ?></div>
            <div class="profile-mobile">📱 <?= htmlspecialchars($user['mobile']) ?></div>
            <span class="profile-status <?= $user['status'] ?>"><?= ucfirst($user['status']) ?></span>
        </div>

        <!-- Stats -->
        <div class="stats-grid">
            <div class="stat-card">
                <div class="stat-value"><?= $bidStats['total'] ?? 0 ?></div>
                <div class="stat-label">Total Bids</div>
            </div>
            <div class="stat-card">
                <div class="stat-value green"><?= $winStats['wins'] ?? 0 ?></div>
                <div class="stat-label">Wins</div>
            </div>
            <div class="stat-card">
                <div class="stat-value">₹<?= number_format($balance, 0) ?></div>
                <div class="stat-label">Balance</div>
            </div>
        </div>

        <!-- Quick Links -->
        <div class="section-card">
            <div class="section-title">⚡ Quick Links</div>
            <div class="quick-links">
                <a href="wallet.php" class="quick-link">
                    <div class="quick-link-left">
                        <div class="quick-link-icon green">💰</div>
                        <div class="quick-link-text">
                            <h4>Wallet</h4>
                            <p>Deposit & Withdraw</p>
                        </div>
                    </div>
                    <span class="quick-link-arrow">→</span>
                </a>
                <a href="history.php" class="quick-link">
                    <div class="quick-link-left">
                        <div class="quick-link-icon blue">📋</div>
                        <div class="quick-link-text">
                            <h4>Bid History</h4>
                            <p>View all your bids</p>
                        </div>
                    </div>
                    <span class="quick-link-arrow">→</span>
                </a>
                <?php 
                $whatsapp = getSetting($pdo, 'whatsapp_number', '');
                $telegram = getSetting($pdo, 'telegram_link', '');
                if ($whatsapp): ?>
                <a href="https://wa.me/<?= $whatsapp ?>" target="_blank" class="quick-link">
                    <div class="quick-link-left">
                        <div class="quick-link-icon" style="background: rgba(37,211,102,0.2);">💬</div>
                        <div class="quick-link-text">
                            <h4>WhatsApp Support</h4>
                            <p>Chat with us</p>
                        </div>
                    </div>
                    <span class="quick-link-arrow">→</span>
                </a>
                <?php endif; ?>
            </div>
        </div>

        <!-- Personal Info -->
        <div class="section-card">
            <div class="section-title">📝 Personal Information</div>
            <form method="POST" enctype="multipart/form-data">
                <input type="hidden" name="csrf_token" value="<?= $csrfToken ?>">
                <div class="file-upload">
                    <div class="file-label">Profile Photo</div>
                    <input type="file" name="profile_photo" id="profilePhoto" class="file-input" accept="image/*">
                    <label for="profilePhoto" class="file-box">
                        <?php if ($avatar && file_exists($avatar)): ?>
                            <img src="<?= $avatar ?>" class="file-preview" alt="">
                        <?php else: ?>
                            <div class="file-placeholder">📷</div>
                        <?php endif; ?>
                        <div class="file-text">
                            <h4>Change Photo</h4>
                            <p>JPG, PNG (Max 2MB)</p>
                        </div>
                    </label>
                </div>
                
                <div class="form-group">
                    <label class="form-label">Full Name</label>
                    <input type="text" name="name" class="form-input" value="<?= htmlspecialchars($user['name']) ?>" required>
                </div>
                
                <div class="form-group">
                    <label class="form-label">Mobile Number</label>
                    <input type="text" class="form-input" value="<?= htmlspecialchars($user['mobile']) ?>" disabled>
                </div>
                
                <button type="submit" name="update_profile" class="btn-submit">Save Changes</button>
            </form>
        </div>

        <!-- Bank Details -->
        <div class="section-card">
            <div class="section-title">🏦 Bank Details</div>
            <form method="POST" enctype="multipart/form-data">
                <input type="hidden" name="csrf_token" value="<?= $csrfToken ?>">
                <input type="hidden" name="name" value="<?= htmlspecialchars($user['name']) ?>">
                
                <div class="form-group">
                    <label class="form-label">Bank Name</label>
                    <input type="text" name="bank_name" class="form-input" value="<?= htmlspecialchars($user['bank_name'] ?? '') ?>" placeholder="e.g. State Bank of India">
                </div>
                
                <div class="form-row">
                    <div class="form-group">
                        <label class="form-label">Account Number</label>
                        <input type="text" name="account_no" class="form-input" value="<?= htmlspecialchars($user['account_no'] ?? '') ?>" placeholder="Account Number">
                    </div>
                    <div class="form-group">
                        <label class="form-label">Account Holder Name</label>
                        <input type="text" name="account_holder" class="form-input" value="<?= htmlspecialchars($user['account_holder'] ?? '') ?>" placeholder="Account Holder Name">
                    </div>
                </div>
                
                <div class="form-row">
                    <div class="form-group">
                        <label class="form-label">IFSC Code</label>
                        <input type="text" name="ifsc_code" class="form-input" value="<?= htmlspecialchars($user['ifsc_code'] ?? '') ?>" placeholder="IFSC Code">
                    </div>
                    <div class="form-group">
                        <label class="form-label">UPI ID</label>
                        <input type="text" name="upi_id" class="form-input" value="<?= htmlspecialchars($user['upi_id'] ?? '') ?>" placeholder="yourname@upi">
                    </div>
                </div>
                
                <div class="file-upload">
                    <div class="file-label">Payment QR Code (Optional)</div>
                    <input type="file" name="qr_code" id="qrCode" class="file-input" accept="image/*">
                    <label for="qrCode" class="file-box">
                        <?php if ($user['qr_code'] && file_exists('uploads/' . $user['qr_code'])): ?>
                            <img src="uploads/<?= $user['qr_code'] ?>" class="file-preview" alt="">
                        <?php else: ?>
                            <div class="file-placeholder">📱</div>
                        <?php endif; ?>
                        <div class="file-text">
                            <h4>Upload QR Code</h4>
                            <p>Your payment QR</p>
                        </div>
                    </label>
                </div>
                
                <button type="submit" name="update_profile" class="btn-submit">Save Bank Details</button>
            </form>
        </div>

        <!-- Change Password -->
        <div class="section-card">
            <div class="section-title">🔒 Change Password</div>
            <form method="POST">
                <input type="hidden" name="csrf_token" value="<?= $csrfToken ?>">
                <div class="form-group">
                    <label class="form-label">Current Password</label>
                    <input type="password" name="current_password" class="form-input" placeholder="Enter current password" required>
                </div>
                
                <div class="form-group">
                    <label class="form-label">New Password</label>
                    <input type="password" name="new_password" class="form-input" placeholder="Enter new password" required>
                </div>
                
                <div class="form-group">
                    <label class="form-label">Confirm New Password</label>
                    <input type="password" name="confirm_password" class="form-input" placeholder="Confirm new password" required>
                </div>
                
                <button type="submit" name="change_password" class="btn-submit red">Change Password</button>
            </form>
        </div>

        <!-- Account Info -->
        <div class="section-card">
            <div class="section-title">ℹ️ Account Info</div>
            <div class="info-row">
                <span class="info-label">Member Since</span>
                <span class="info-value"><?= date('d M Y', strtotime($user['created_at'])) ?></span>
            </div>
            <div class="info-row">
                <span class="info-label">Total Played</span>
                <span class="info-value">₹<?= number_format($bidStats['total_amount'] ?? 0, 0) ?></span>
            </div>
            <div class="info-row">
                <span class="info-label">Total Won</span>
                <span class="info-value" style="color: #00ff88;">₹<?= number_format($winStats['win_amount'] ?? 0, 0) ?></span>
            </div>
        </div>

        <!-- Logout -->
        <a href="logout.php" class="btn-submit red" style="display: block; text-align: center; text-decoration: none; margin-top: 20px;">
            🚪 Logout
        </a>
    </div>

    <nav class="bottom-nav">
        <a href="index.php" class="nav-item">
            <span class="nav-icon">🏠</span>
            <span class="nav-label">Home</span>
        </a>
        <a href="wallet.php" class="nav-item">
            <span class="nav-icon">💰</span>
            <span class="nav-label">Wallet</span>
        </a>
        <a href="history.php" class="nav-item">
            <span class="nav-icon">📋</span>
            <span class="nav-label">History</span>
        </a>
        <a href="profile.php" class="nav-item active">
            <span class="nav-icon">👤</span>
            <span class="nav-label">Profile</span>
        </a>
    </nav>
</body>
</html>
