<?php
require_once 'config/db.php';
require_once 'config/auth.php';
require_once 'config/functions.php';
require_once 'config/security.php';

initSecureSession();
requireLogin();

if (!checkSessionTimeout($pdo)) {
    header("Location: login.php?msg=Session+expired");
    exit();
}

$user = getUser($pdo);
if (!$user) {
    session_destroy();
    header("Location: login.php");
    exit();
}

trackUserSession($pdo, $user['id']);

$balance = getWalletBalance($pdo, $user['id']);

// Get all bids with game info (with archive support)
$stmt = $pdo->prepare("
    SELECT b.*, 
    COALESCE(g.name, b.game_name_archive, CONCAT('Game #', b.game_id)) as game_name, 
    COALESCE(g.rate, 95) as game_rate
    FROM bids b 
    LEFT JOIN games g ON b.game_id = g.id 
    WHERE b.user_id = ? 
    ORDER BY b.created_at DESC 
    LIMIT 100
");
$stmt->execute([$user['id']]);
$bids = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Calculate stats
$totalBids = count($bids);
$totalBidAmount = 0;
$totalWins = 0;
$totalWinAmount = 0;
$totalLoss = 0;

foreach ($bids as $bid) {
    $totalBidAmount += $bid['amount'];
    if ($bid['status'] == 'won') {
        $totalWins++;
        $totalWinAmount += $bid['win_amount'];
    } elseif ($bid['status'] == 'lost') {
        $totalLoss++;
    }
}

$profitLoss = $totalWinAmount - $totalBidAmount;
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>My History - Satta King</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
            background: #0f0f0f;
            color: #fff;
            min-height: 100vh;
            padding-bottom: 80px;
        }
        
        /* Header */
        .header {
            background: linear-gradient(135deg, #8b0000 0%, #1a1a1a 100%);
            padding: 15px 20px;
            display: flex;
            justify-content: space-between;
            align-items: center;
            position: sticky;
            top: 0;
            z-index: 100;
        }
        
        .back-btn {
            color: #fff;
            text-decoration: none;
            font-size: 16px;
        }
        
        .page-title {
            font-size: 18px;
            font-weight: 700;
            color: #ffd700;
        }
        
        .wallet-display {
            background: rgba(255,215,0,0.2);
            padding: 8px 15px;
            border-radius: 20px;
            font-size: 14px;
            font-weight: 600;
            color: #ffd700;
        }
        
        /* Container */
        .container {
            padding: 20px;
            max-width: 600px;
            margin: 0 auto;
        }
        
        /* Stats Grid */
        .stats-grid {
            display: grid;
            grid-template-columns: repeat(2, 1fr);
            gap: 12px;
            margin-bottom: 25px;
        }
        
        .stat-card {
            background: linear-gradient(145deg, #1e1e1e 0%, #2a2a2a 100%);
            border-radius: 15px;
            padding: 20px;
            text-align: center;
            border: 1px solid #333;
        }
        
        .stat-icon {
            font-size: 28px;
            margin-bottom: 8px;
        }
        
        .stat-value {
            font-size: 24px;
            font-weight: 700;
            color: #ffd700;
        }
        
        .stat-value.green {
            color: #00ff88;
        }
        
        .stat-value.red {
            color: #ff4444;
        }
        
        .stat-label {
            font-size: 12px;
            color: #888;
            margin-top: 5px;
        }
        
        /* Profit/Loss Card */
        .profit-card {
            background: linear-gradient(135deg, #1a1a2e 0%, #16213e 100%);
            border-radius: 15px;
            padding: 25px;
            text-align: center;
            margin-bottom: 25px;
            border: 1px solid #333;
        }
        
        .profit-label {
            font-size: 14px;
            color: #888;
            margin-bottom: 5px;
        }
        
        .profit-amount {
            font-size: 36px;
            font-weight: 800;
        }
        
        .profit-amount.profit {
            color: #00ff88;
        }
        
        .profit-amount.loss {
            color: #ff4444;
        }
        
        /* Section Title */
        .section-title {
            color: #ffd700;
            font-size: 18px;
            margin-bottom: 20px;
            padding-bottom: 10px;
            border-bottom: 2px solid #333;
        }
        
        /* Filter Tabs */
        .filter-tabs {
            display: flex;
            gap: 10px;
            margin-bottom: 20px;
            overflow-x: auto;
            padding-bottom: 5px;
        }
        
        .filter-tab {
            padding: 10px 20px;
            background: #1a1a1a;
            border: 1px solid #333;
            border-radius: 25px;
            color: #888;
            font-size: 13px;
            font-weight: 600;
            cursor: pointer;
            white-space: nowrap;
            transition: all 0.3s;
        }
        
        .filter-tab.active {
            background: #ffd700;
            color: #000;
            border-color: #ffd700;
        }
        
        /* Bid Card */
        .bid-card {
            background: linear-gradient(145deg, #1e1e1e 0%, #2a2a2a 100%);
            border-radius: 15px;
            margin-bottom: 15px;
            overflow: hidden;
            border: 1px solid #333;
        }
        
        .bid-card.won {
            border-left: 4px solid #00ff88;
        }
        
        .bid-card.lost {
            border-left: 4px solid #ff4444;
        }
        
        .bid-card.pending {
            border-left: 4px solid #ffd700;
        }
        
        .bid-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding: 15px;
            background: rgba(0,0,0,0.3);
        }
        
        .bid-game {
            display: flex;
            align-items: center;
            gap: 10px;
        }
        
        .bid-game-icon {
            width: 40px;
            height: 40px;
            background: linear-gradient(135deg, #8b0000 0%, #5c0000 100%);
            border-radius: 10px;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 18px;
        }
        
        .bid-game-info h4 {
            font-size: 14px;
            font-weight: 600;
            margin-bottom: 3px;
        }
        
        .bid-game-info p {
            font-size: 11px;
            color: #888;
        }
        
        .bid-status {
            padding: 6px 12px;
            border-radius: 20px;
            font-size: 11px;
            font-weight: 700;
            text-transform: uppercase;
        }
        
        .bid-status.won {
            background: rgba(0,255,136,0.2);
            color: #00ff88;
        }
        
        .bid-status.lost {
            background: rgba(255,68,68,0.2);
            color: #ff4444;
        }
        
        .bid-status.pending {
            background: rgba(255,215,0,0.2);
            color: #ffd700;
        }
        
        .bid-body {
            padding: 15px;
        }
        
        .bid-details {
            display: grid;
            grid-template-columns: repeat(3, 1fr);
            gap: 15px;
            margin-bottom: 15px;
        }
        
        .bid-detail {
            text-align: center;
        }
        
        .bid-detail-label {
            font-size: 10px;
            color: #666;
            text-transform: uppercase;
            margin-bottom: 5px;
        }
        
        .bid-detail-value {
            font-size: 16px;
            font-weight: 700;
        }
        
        .bid-detail-value.type-jodi {
            color: #3498db;
        }
        
        .bid-detail-value.type-andar {
            color: #2ecc71;
        }
        
        .bid-detail-value.type-bahar {
            color: #e74c3c;
        }
        
        .bid-detail-value.number {
            color: #ffd700;
            font-size: 22px;
        }
        
        .bid-footer {
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding-top: 15px;
            border-top: 1px solid #333;
        }
        
        .bid-amount-info {
            display: flex;
            gap: 20px;
        }
        
        .amount-item {
            text-align: center;
        }
        
        .amount-label {
            font-size: 10px;
            color: #666;
        }
        
        .amount-value {
            font-size: 16px;
            font-weight: 700;
        }
        
        .amount-value.debit {
            color: #ff4444;
        }
        
        .amount-value.credit {
            color: #00ff88;
        }
        
        .bid-result {
            text-align: right;
        }
        
        .result-label {
            font-size: 10px;
            color: #666;
        }
        
        .result-value {
            font-size: 22px;
            font-weight: 800;
            color: #00ff88;
        }
        
        .result-waiting {
            font-size: 16px;
            color: #666;
        }
        
        /* Empty State */
        .empty-state {
            text-align: center;
            padding: 60px 20px;
            color: #666;
        }
        
        .empty-icon {
            font-size: 60px;
            margin-bottom: 15px;
        }
        
        .empty-text {
            font-size: 16px;
            margin-bottom: 20px;
        }
        
        .btn-play {
            display: inline-block;
            background: linear-gradient(135deg, #00c853 0%, #00a844 100%);
            color: #fff;
            padding: 12px 30px;
            border-radius: 25px;
            text-decoration: none;
            font-weight: 700;
        }
        
        /* Bottom Navigation */
        .bottom-nav {
            position: fixed;
            bottom: 0;
            left: 0;
            right: 0;
            background: #1a1a1a;
            display: flex;
            justify-content: space-around;
            padding: 10px 0;
            border-top: 1px solid #333;
            z-index: 100;
        }
        
        .nav-item {
            text-align: center;
            color: #666;
            text-decoration: none;
            padding: 5px 15px;
        }
        
        .nav-item.active {
            color: #ffd700;
        }
        
        .nav-icon {
            font-size: 22px;
            display: block;
            margin-bottom: 4px;
        }
        
        .nav-label {
            font-size: 10px;
        }
        
        /* Responsive */
        @media (max-width: 400px) {
            .stats-grid {
                grid-template-columns: repeat(2, 1fr);
                gap: 10px;
            }
            .stat-card {
                padding: 15px;
            }
            .stat-value {
                font-size: 20px;
            }
            .bid-details {
                grid-template-columns: repeat(3, 1fr);
                gap: 10px;
            }
            .profit-amount {
                font-size: 28px;
            }
        }
    </style>
</head>
<body>
    <header class="header">
        <a href="index.php" class="back-btn">← Back</a>
        <div class="page-title">📋 My History</div>
        <div class="wallet-display">₹<?= number_format($balance, 2) ?></div>
    </header>

    <div class="container">
        <!-- Stats Grid -->
        <div class="stats-grid">
            <div class="stat-card">
                <div class="stat-icon">🎯</div>
                <div class="stat-value"><?= $totalBids ?></div>
                <div class="stat-label">Total Bids</div>
            </div>
            <div class="stat-card">
                <div class="stat-icon">💰</div>
                <div class="stat-value">₹<?= number_format($totalBidAmount, 0) ?></div>
                <div class="stat-label">Total Played</div>
            </div>
            <div class="stat-card">
                <div class="stat-icon">🏆</div>
                <div class="stat-value green"><?= $totalWins ?></div>
                <div class="stat-label">Wins</div>
            </div>
            <div class="stat-card">
                <div class="stat-icon">❌</div>
                <div class="stat-value red"><?= $totalLoss ?></div>
                <div class="stat-label">Loss</div>
            </div>
        </div>

        <!-- Profit/Loss Card -->
        <div class="profit-card">
            <div class="profit-label">Total Profit / Loss</div>
            <div class="profit-amount <?= $profitLoss >= 0 ? 'profit' : 'loss' ?>">
                <?= $profitLoss >= 0 ? '+' : '' ?>₹<?= number_format($profitLoss, 0) ?>
            </div>
        </div>

        <!-- Filter Tabs -->
        <div class="filter-tabs">
            <div class="filter-tab active" onclick="filterBids('all')">All</div>
            <div class="filter-tab" onclick="filterBids('won')">Won</div>
            <div class="filter-tab" onclick="filterBids('lost')">Lost</div>
            <div class="filter-tab" onclick="filterBids('pending')">Pending</div>
            <div class="filter-tab" onclick="filterBids('jodi')">Jodi</div>
            <div class="filter-tab" onclick="filterBids('andar')">Andar</div>
            <div class="filter-tab" onclick="filterBids('bahar')">Bahar</div>
        </div>

        <h2 class="section-title">📝 Bid History</h2>

        <?php if (empty($bids)): ?>
            <div class="empty-state">
                <div class="empty-icon">📋</div>
                <div class="empty-text">No bids placed yet</div>
                <a href="index.php" class="btn-play">▶ Play Now</a>
            </div>
        <?php else: ?>
            <div id="bidsList">
                <?php foreach ($bids as $bid): ?>
                    <div class="bid-card <?= $bid['status'] ?>" data-status="<?= $bid['status'] ?>" data-type="<?= $bid['bid_type'] ?>">
                        <div class="bid-header">
                            <div class="bid-game">
                                <div class="bid-game-icon">🎰</div>
                                <div class="bid-game-info">
                                    <h4><?= htmlspecialchars($bid['game_name']) ?></h4>
                                    <p><?= date('d M Y, h:i A', strtotime($bid['created_at'])) ?></p>
                                </div>
                            </div>
                            <div class="bid-status <?= $bid['status'] ?>">
                                <?php if ($bid['status'] == 'won'): ?>
                                    🏆 WON
                                <?php elseif ($bid['status'] == 'lost'): ?>
                                    ❌ LOST
                                <?php else: ?>
                                    ⏳ PENDING
                                <?php endif; ?>
                            </div>
                        </div>
                        
                        <div class="bid-body">
                            <div class="bid-details">
                                <div class="bid-detail">
                                    <div class="bid-detail-label">Type</div>
                                    <div class="bid-detail-value type-<?= $bid['bid_type'] ?>">
                                        <?= strtoupper($bid['bid_type']) ?>
                                    </div>
                                </div>
                                <div class="bid-detail">
                                    <div class="bid-detail-label">Number</div>
                                    <div class="bid-detail-value number"><?= $bid['number'] ?></div>
                                </div>
                                <div class="bid-detail">
                                    <div class="bid-detail-label">Date</div>
                                    <div class="bid-detail-value"><?= date('d M', strtotime($bid['playing_date'])) ?></div>
                                </div>
                            </div>
                            
                            <div class="bid-footer">
                                <div class="bid-amount-info">
                                    <div class="amount-item">
                                        <div class="amount-label">Bid Amount</div>
                                        <div class="amount-value debit">-₹<?= number_format($bid['amount'], 0) ?></div>
                                    </div>
                                    <?php if ($bid['status'] == 'won'): ?>
                                        <div class="amount-item">
                                            <div class="amount-label">Won Amount</div>
                                            <div class="amount-value credit">+₹<?= number_format($bid['win_amount'], 0) ?></div>
                                        </div>
                                    <?php endif; ?>
                                </div>
                                
                                <?php if ($bid['status'] == 'won'): ?>
                                    <div class="bid-result">
                                        <div class="result-label">Profit</div>
                                        <div class="result-value">+₹<?= number_format($bid['win_amount'] - $bid['amount'], 0) ?></div>
                                    </div>
                                <?php elseif ($bid['status'] == 'pending'): ?>
                                    <div class="bid-result">
                                        <div class="result-waiting">Waiting for result...</div>
                                    </div>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>
                <?php endforeach; ?>
            </div>
        <?php endif; ?>
    </div>

    <nav class="bottom-nav">
        <a href="index.php" class="nav-item">
            <span class="nav-icon">🏠</span>
            <span class="nav-label">Home</span>
        </a>
        <a href="wallet.php" class="nav-item">
            <span class="nav-icon">💰</span>
            <span class="nav-label">Wallet</span>
        </a>
        <a href="history.php" class="nav-item active">
            <span class="nav-icon">📋</span>
            <span class="nav-label">History</span>
        </a>
        <a href="profile.php" class="nav-item">
            <span class="nav-icon">👤</span>
            <span class="nav-label">Profile</span>
        </a>
    </nav>

    <script>
        function filterBids(filter) {
            // Update active tab
            document.querySelectorAll('.filter-tab').forEach(tab => {
                tab.classList.remove('active');
            });
            event.target.classList.add('active');
            
            // Filter cards
            document.querySelectorAll('.bid-card').forEach(card => {
                const status = card.dataset.status;
                const type = card.dataset.type;
                
                if (filter === 'all') {
                    card.style.display = 'block';
                } else if (filter === 'won' || filter === 'lost' || filter === 'pending') {
                    card.style.display = status === filter ? 'block' : 'none';
                } else if (filter === 'jodi' || filter === 'andar' || filter === 'bahar') {
                    card.style.display = type === filter ? 'block' : 'none';
                }
            });
        }
    </script>
</body>
</html>
