-- =============================================
-- SATTA KING - DATABASE UPGRADE
-- Security & Feature Improvements
-- Run this on your existing database
-- =============================================

-- =============================================
-- 1. LOGIN ATTEMPTS TABLE (Brute Force Protection)
-- =============================================
CREATE TABLE IF NOT EXISTS `login_attempts` (
  `id` int(11) NOT NULL AUTO_INCREMENT,
  `ip_address` varchar(50) NOT NULL,
  `mobile` varchar(15) DEFAULT NULL,
  `attempt_time` timestamp DEFAULT current_timestamp(),
  `success` tinyint(1) DEFAULT 0,
  PRIMARY KEY (`id`),
  KEY `idx_ip_time` (`ip_address`, `attempt_time`),
  KEY `idx_mobile_time` (`mobile`, `attempt_time`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- =============================================
-- 2. CSRF TOKENS TABLE
-- =============================================
CREATE TABLE IF NOT EXISTS `csrf_tokens` (
  `id` int(11) NOT NULL AUTO_INCREMENT,
  `token` varchar(64) NOT NULL,
  `user_id` int(11) DEFAULT NULL,
  `session_id` varchar(128) NOT NULL,
  `created_at` timestamp DEFAULT current_timestamp(),
  `expires_at` timestamp NOT NULL,
  PRIMARY KEY (`id`),
  UNIQUE KEY `token` (`token`),
  KEY `idx_session` (`session_id`),
  KEY `idx_expires` (`expires_at`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- =============================================
-- 3. USER SESSIONS TABLE (Multiple Device Tracking)
-- =============================================
CREATE TABLE IF NOT EXISTS `user_sessions` (
  `id` int(11) NOT NULL AUTO_INCREMENT,
  `user_id` int(11) NOT NULL,
  `session_id` varchar(128) NOT NULL,
  `ip_address` varchar(50) DEFAULT NULL,
  `user_agent` varchar(255) DEFAULT NULL,
  `device_info` varchar(100) DEFAULT NULL,
  `last_activity` timestamp DEFAULT current_timestamp() ON UPDATE current_timestamp(),
  `created_at` timestamp DEFAULT current_timestamp(),
  `is_active` tinyint(1) DEFAULT 1,
  PRIMARY KEY (`id`),
  UNIQUE KEY `session_id` (`session_id`),
  KEY `idx_user` (`user_id`),
  KEY `idx_active` (`is_active`, `last_activity`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- =============================================
-- 4. RATE LIMITING TABLE
-- =============================================
CREATE TABLE IF NOT EXISTS `rate_limits` (
  `id` int(11) NOT NULL AUTO_INCREMENT,
  `identifier` varchar(100) NOT NULL,
  `action_type` varchar(50) NOT NULL,
  `count` int(11) DEFAULT 1,
  `window_start` timestamp DEFAULT current_timestamp(),
  PRIMARY KEY (`id`),
  UNIQUE KEY `idx_identifier_action` (`identifier`, `action_type`),
  KEY `idx_window` (`window_start`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- =============================================
-- 5. NOTIFICATIONS TABLE
-- =============================================
CREATE TABLE IF NOT EXISTS `notifications` (
  `id` int(11) NOT NULL AUTO_INCREMENT,
  `user_id` int(11) NOT NULL,
  `type` varchar(50) NOT NULL,
  `title` varchar(200) NOT NULL,
  `message` text NOT NULL,
  `data` longtext DEFAULT NULL,
  `is_read` tinyint(1) DEFAULT 0,
  `created_at` timestamp DEFAULT current_timestamp(),
  PRIMARY KEY (`id`),
  KEY `idx_user_read` (`user_id`, `is_read`),
  KEY `idx_created` (`created_at`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- =============================================
-- 6. REFERRALS TABLE
-- =============================================
CREATE TABLE IF NOT EXISTS `referrals` (
  `id` int(11) NOT NULL AUTO_INCREMENT,
  `referrer_id` int(11) NOT NULL,
  `referred_id` int(11) NOT NULL,
  `referral_code` varchar(20) NOT NULL,
  `bonus_given` tinyint(1) DEFAULT 0,
  `referrer_bonus` decimal(10,2) DEFAULT 0.00,
  `referred_bonus` decimal(10,2) DEFAULT 0.00,
  `created_at` timestamp DEFAULT current_timestamp(),
  PRIMARY KEY (`id`),
  UNIQUE KEY `referred_id` (`referred_id`),
  KEY `idx_referrer` (`referrer_id`),
  KEY `idx_code` (`referral_code`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- =============================================
-- 7. ADMIN ROLES TABLE
-- =============================================
CREATE TABLE IF NOT EXISTS `admin_roles` (
  `id` int(11) NOT NULL AUTO_INCREMENT,
  `name` varchar(50) NOT NULL,
  `description` varchar(200) DEFAULT NULL,
  `permissions` longtext DEFAULT NULL,
  `created_at` timestamp DEFAULT current_timestamp(),
  PRIMARY KEY (`id`),
  UNIQUE KEY `name` (`name`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Insert default admin roles
INSERT INTO `admin_roles` (`name`, `description`, `permissions`) VALUES
('super_admin', 'Full Access to Everything', '{"all": true}'),
('finance_admin', 'Wallet & Transaction Management', '{"wallet": true, "reports": true}'),
('game_admin', 'Game & Result Management', '{"games": true, "results": true, "bids": true}'),
('support', 'View Only Access', '{"view_users": true, "view_bids": true}');

-- =============================================
-- 8. ADD NEW COLUMNS TO USERS TABLE
-- =============================================
ALTER TABLE `users` 
ADD COLUMN IF NOT EXISTS `transaction_pin` varchar(255) DEFAULT NULL AFTER `password`,
ADD COLUMN IF NOT EXISTS `referral_code` varchar(20) DEFAULT NULL AFTER `transaction_pin`,
ADD COLUMN IF NOT EXISTS `referred_by` int(11) DEFAULT NULL AFTER `referral_code`,
ADD COLUMN IF NOT EXISTS `admin_role_id` int(11) DEFAULT NULL AFTER `is_admin`,
ADD COLUMN IF NOT EXISTS `daily_withdraw_total` decimal(10,2) DEFAULT 0.00 AFTER `admin_role_id`,
ADD COLUMN IF NOT EXISTS `daily_withdraw_date` date DEFAULT NULL AFTER `daily_withdraw_total`,
ADD COLUMN IF NOT EXISTS `daily_bid_total` decimal(10,2) DEFAULT 0.00 AFTER `daily_withdraw_date`,
ADD COLUMN IF NOT EXISTS `daily_bid_date` date DEFAULT NULL AFTER `daily_bid_total`,
ADD COLUMN IF NOT EXISTS `failed_login_attempts` int(11) DEFAULT 0 AFTER `daily_bid_date`,
ADD COLUMN IF NOT EXISTS `locked_until` timestamp NULL DEFAULT NULL AFTER `failed_login_attempts`,
ADD COLUMN IF NOT EXISTS `last_login` timestamp NULL DEFAULT NULL AFTER `locked_until`,
ADD COLUMN IF NOT EXISTS `last_login_ip` varchar(50) DEFAULT NULL AFTER `last_login`;

-- Add unique index for referral code
ALTER TABLE `users` ADD UNIQUE INDEX IF NOT EXISTS `idx_referral_code` (`referral_code`);

-- =============================================
-- 9. ADD NEW COLUMNS TO BIDS TABLE (Cancellation)
-- =============================================
ALTER TABLE `bids` 
ADD COLUMN IF NOT EXISTS `is_cancelled` tinyint(1) DEFAULT 0 AFTER `status`,
ADD COLUMN IF NOT EXISTS `cancelled_at` timestamp NULL DEFAULT NULL AFTER `is_cancelled`,
ADD COLUMN IF NOT EXISTS `refund_processed` tinyint(1) DEFAULT 0 AFTER `cancelled_at`;

-- =============================================
-- 10. NEW SETTINGS FOR LIMITS
-- =============================================
INSERT IGNORE INTO `settings` (`setting_key`, `setting_value`, `setting_type`, `description`) VALUES
('daily_withdraw_limit', '50000', 'number', 'Maximum withdrawal per day per user'),
('weekly_withdraw_limit', '200000', 'number', 'Maximum withdrawal per week per user'),
('daily_bid_limit', '100000', 'number', 'Maximum bid amount per day per user'),
('bid_cancel_minutes', '5', 'number', 'Minutes before close time to allow bid cancellation'),
('session_timeout_minutes', '60', 'number', 'Session timeout in minutes'),
('max_login_attempts', '5', 'number', 'Maximum failed login attempts before lockout'),
('lockout_duration_minutes', '30', 'number', 'Account lockout duration in minutes'),
('referral_bonus_referrer', '50', 'number', 'Bonus amount for referrer'),
('referral_bonus_referred', '25', 'number', 'Bonus amount for new user'),
('require_transaction_pin', '1', 'boolean', 'Require PIN for withdrawals');

-- =============================================
-- 11. DAILY REPORTS TABLE
-- =============================================
CREATE TABLE IF NOT EXISTS `daily_reports` (
  `id` int(11) NOT NULL AUTO_INCREMENT,
  `report_date` date NOT NULL,
  `total_users` int(11) DEFAULT 0,
  `new_users` int(11) DEFAULT 0,
  `total_bids` int(11) DEFAULT 0,
  `total_bid_amount` decimal(12,2) DEFAULT 0.00,
  `total_wins` int(11) DEFAULT 0,
  `total_payout` decimal(12,2) DEFAULT 0.00,
  `total_deposits` decimal(12,2) DEFAULT 0.00,
  `total_withdrawals` decimal(12,2) DEFAULT 0.00,
  `gross_profit` decimal(12,2) DEFAULT 0.00,
  `created_at` timestamp DEFAULT current_timestamp(),
  PRIMARY KEY (`id`),
  UNIQUE KEY `report_date` (`report_date`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- =============================================
-- 12. SUSPICIOUS ACTIVITY LOG
-- =============================================
CREATE TABLE IF NOT EXISTS `security_log` (
  `id` int(11) NOT NULL AUTO_INCREMENT,
  `user_id` int(11) DEFAULT NULL,
  `event_type` varchar(50) NOT NULL,
  `severity` enum('low','medium','high','critical') DEFAULT 'low',
  `ip_address` varchar(50) DEFAULT NULL,
  `user_agent` varchar(255) DEFAULT NULL,
  `details` text DEFAULT NULL,
  `created_at` timestamp DEFAULT current_timestamp(),
  PRIMARY KEY (`id`),
  KEY `idx_user` (`user_id`),
  KEY `idx_type` (`event_type`),
  KEY `idx_severity` (`severity`),
  KEY `idx_created` (`created_at`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- =============================================
-- 13. CLEANUP OLD DATA (Run periodically)
-- =============================================
-- Delete old login attempts (older than 24 hours)
-- DELETE FROM login_attempts WHERE attempt_time < DATE_SUB(NOW(), INTERVAL 24 HOUR);

-- Delete expired CSRF tokens
-- DELETE FROM csrf_tokens WHERE expires_at < NOW();

-- Delete old rate limit records (older than 1 hour)
-- DELETE FROM rate_limits WHERE window_start < DATE_SUB(NOW(), INTERVAL 1 HOUR);

-- =============================================
-- UPDATE EXISTING USERS WITH REFERRAL CODES
-- =============================================
-- Generate unique referral codes for existing users
UPDATE `users` SET `referral_code` = UPPER(CONCAT(
    SUBSTRING(MD5(RAND()), 1, 4),
    id
)) WHERE `referral_code` IS NULL AND `is_admin` = 0;

-- Set super_admin role for existing admin
UPDATE `users` SET `admin_role_id` = 1 WHERE `is_admin` = 1 AND `admin_role_id` IS NULL;
