<?php
/**
 * Authentication & Game Logic Functions
 * Satta King Platform
 */

// Set timezone to India
date_default_timezone_set('Asia/Kolkata');

if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

// =============================================
// AUTHENTICATION FUNCTIONS
// =============================================

function isLoggedIn() {
    return isset($_SESSION['user_id']);
}

function isAdmin() {
    return isset($_SESSION['is_admin']) && $_SESSION['is_admin'] == 1;
}

function requireLogin() {
    if (!isLoggedIn()) {
        header("Location: login.php");
        exit();
    }
}

function requireAdmin() {
    if (!isAdmin()) {
        header("Location: ../login.php");
        exit();
    }
}

function getUser($pdo) {
    if (!isLoggedIn()) return null;
    $stmt = $pdo->prepare("SELECT * FROM users WHERE id = ?");
    $stmt->execute([$_SESSION['user_id']]);
    return $stmt->fetch(PDO::FETCH_ASSOC);
}

function getWalletBalance($pdo, $user_id) {
    $stmt = $pdo->prepare("SELECT 
        COALESCE(SUM(CASE WHEN type = 'credit' THEN amount ELSE 0 END), 0) -
        COALESCE(SUM(CASE WHEN type = 'debit' THEN amount ELSE 0 END), 0) as balance 
        FROM wallet WHERE user_id = ?");
    $stmt->execute([$user_id]);
    $result = $stmt->fetch(PDO::FETCH_ASSOC);
    return $result['balance'] ?? 0;
}

function hasPendingRequest($pdo, $user_id, $type = null) {
    $sql = "SELECT COUNT(*) as count FROM wallet_requests WHERE user_id = ? AND status = 'pending'";
    $params = [$user_id];
    
    if ($type) {
        $sql .= " AND type = ?";
        $params[] = $type;
    }
    
    $stmt = $pdo->prepare($sql);
    $stmt->execute($params);
    return $stmt->fetch()['count'] > 0;
}

// =============================================
// TIME HELPER FUNCTIONS
// =============================================

/**
 * Convert time string to minutes for easy comparison
 */
function timeToMinutes($time) {
    $parts = explode(':', $time);
    return intval($parts[0]) * 60 + intval($parts[1]);
}

/**
 * Check if today is last day of month
 */
function isLastDayOfMonth() {
    $today = date('j');
    $lastDay = date('t');
    return $today == $lastDay;
}

/**
 * Check if today is first day of month
 */
function isFirstDayOfMonth() {
    return date('j') == 1;
}

/**
 * Check if today is second day of month
 */
function isSecondDayOfMonth() {
    return date('j') == 2;
}

// =============================================
// GAME LOGIC
// =============================================
/*
|--------------------------------------------------------------------------
| NORMAL GAME LOGIC (Close 20:00)
|--------------------------------------------------------------------------
|
| Last Day (30 Nov): HOLIDAY
| 1st Day (1 Dec) 00:00-20:00: OPEN (Playing: 1 Dec, no result check)
| 1st Day (1 Dec) 20:00-23:59: CLOSED
| 2nd Day onwards: Normal logic (need previous day result)
|
|--------------------------------------------------------------------------
| OVERNIGHT GAME LOGIC (Open 05:00, Close 03:00)
|--------------------------------------------------------------------------
|
| Last Day (30 Nov): HOLIDAY
|
| 1st Day (1 Dec) 00:00:00 ─────────────────► 2nd Day (2 Dec) 03:00:00
|     NO GAP! Continuous bidding (Playing: 2 Dec, no result check)
|
| 2nd Day (2 Dec) 03:00:00 ─────────────────► Result aane tak
|     GAP / RESULT PENDING (waiting for 2 Dec result)
|
| 2nd Day (2 Dec) Result aate hi ───────────► 3rd Day (3 Dec) 03:00:00
|     OPEN (Playing: 3 Dec, need 2 Dec result)
|
| Normal flow continues...
|
*/

/**
 * NORMAL GAME - Check if time window is open
 */
function isNormalGameTimeOpen($game) {
    if (isLastDayOfMonth()) {
        return false;
    }
    
    $currentMinutes = timeToMinutes(date('H:i'));
    $closeMinutes = timeToMinutes($game['close_time']);
    
    return $currentMinutes < $closeMinutes;
}

/**
 * NORMAL GAME - Get playing date (always TODAY)
 */
function getNormalGamePlayingDate() {
    return date('Y-m-d');
}

/**
 * NORMAL GAME - Get required result date (YESTERDAY)
 */
function getNormalGameResultDate() {
    $yesterday = new DateTime('-1 day');
    return $yesterday->format('Y-m-d');
}

/**
 * OVERNIGHT GAME - Check if time window is open
 */
function isOvernightGameTimeOpen($game) {
    if (isLastDayOfMonth()) {
        return false;
    }
    
    $currentMinutes = timeToMinutes(date('H:i'));
    $closeMinutes = timeToMinutes($game['close_time']);
    $openMinutes = timeToMinutes($game['open_time']);
    
    // 1st day = NO GAP, always open
    if (isFirstDayOfMonth()) {
        return true;
    }
    
    // 2nd day before close_time = Still first game, NO GAP
    if (isSecondDayOfMonth() && $currentMinutes < $closeMinutes) {
        return true;
    }
    
    // Normal days: Gap period = close_time to open_time (e.g., 03:00 to 05:00)
    if ($currentMinutes >= $closeMinutes && $currentMinutes < $openMinutes) {
        return false;
    }
    
    return true;
}

/**
 * OVERNIGHT GAME - Get playing date
 */
function getOvernightGamePlayingDate($game) {
    $currentMinutes = timeToMinutes(date('H:i'));
    $closeMinutes = timeToMinutes($game['close_time']);
    $openMinutes = timeToMinutes($game['open_time']);
    
    // 1st day = Playing for 2nd day (tomorrow)
    if (isFirstDayOfMonth()) {
        $tomorrow = new DateTime('+1 day');
        return $tomorrow->format('Y-m-d');
    }
    
    // Early morning (00:00 to close_time) - Playing for TODAY
    if ($currentMinutes < $closeMinutes) {
        return date('Y-m-d');
    }
    
    // After open_time - Playing for TOMORROW
    if ($currentMinutes >= $openMinutes) {
        $tomorrow = new DateTime('+1 day');
        return $tomorrow->format('Y-m-d');
    }
    
    // Gap period
    return date('Y-m-d');
}

/**
 * OVERNIGHT GAME - Get required result date
 */
function getOvernightGameResultDate($game) {
    $currentMinutes = timeToMinutes(date('H:i'));
    $closeMinutes = timeToMinutes($game['close_time']);
    
    // Early morning (00:00 to close_time) - Need YESTERDAY's result
    if ($currentMinutes < $closeMinutes) {
        $yesterday = new DateTime('-1 day');
        return $yesterday->format('Y-m-d');
    }
    
    // After close_time - Need TODAY's result
    return date('Y-m-d');
}

/**
 * Check if result check should be skipped (fresh month start)
 */
function shouldSkipResultCheck($game) {
    if ($game['is_overnight']) {
        if (isFirstDayOfMonth()) {
            return true;
        }
        if (isSecondDayOfMonth()) {
            $currentMinutes = timeToMinutes(date('H:i'));
            $closeMinutes = timeToMinutes($game['close_time']);
            if ($currentMinutes < $closeMinutes) {
                return true;
            }
        }
    } else {
        if (isFirstDayOfMonth()) {
            return true;
        }
    }
    return false;
}

/**
 * MAIN FUNCTION - Get Playing Date for bids
 */
function getGamePlayingDate($game) {
    if ($game['is_overnight']) {
        return getOvernightGamePlayingDate($game);
    } else {
        return getNormalGamePlayingDate();
    }
}

/**
 * MAIN FUNCTION - Check if game is open
 */
function isGameOpen($pdo, $game) {
    if ($game['is_overnight']) {
        if (!isOvernightGameTimeOpen($game)) {
            return false;
        }
        
        if (shouldSkipResultCheck($game)) {
            return true;
        }
        
        $resultDate = getOvernightGameResultDate($game);
        $stmt = $pdo->prepare("SELECT id FROM results WHERE game_id = ? AND playing_date = ? LIMIT 1");
        $stmt->execute([$game['id'], $resultDate]);
        
        return $stmt->fetch() ? true : false;
        
    } else {
        if (!isNormalGameTimeOpen($game)) {
            return false;
        }
        
        if (shouldSkipResultCheck($game)) {
            return true;
        }
        
        $resultDate = getNormalGameResultDate();
        $stmt = $pdo->prepare("SELECT id FROM results WHERE game_id = ? AND playing_date = ? LIMIT 1");
        $stmt->execute([$game['id'], $resultDate]);
        
        return $stmt->fetch() ? true : false;
    }
}

/**
 * MAIN FUNCTION - Get game status text
 */
function getGameStatusText($pdo, $game) {
    if (isLastDayOfMonth()) {
        return "Holiday";
    }
    
    if ($game['is_overnight']) {
        $currentMinutes = timeToMinutes(date('H:i'));
        $closeMinutes = timeToMinutes($game['close_time']);
        $openMinutes = timeToMinutes($game['open_time']);
        
        if (isFirstDayOfMonth()) {
            return "Open";
        }
        
        if (isSecondDayOfMonth() && $currentMinutes < $closeMinutes) {
            return "Open";
        }
        
        if ($currentMinutes >= $closeMinutes && $currentMinutes < $openMinutes) {
            $resultDate = getOvernightGameResultDate($game);
            $stmt = $pdo->prepare("SELECT id FROM results WHERE game_id = ? AND playing_date = ? LIMIT 1");
            $stmt->execute([$game['id'], $resultDate]);
            return $stmt->fetch() ? "Closed" : "Result Pending";
        }
        
        $resultDate = getOvernightGameResultDate($game);
        $stmt = $pdo->prepare("SELECT id FROM results WHERE game_id = ? AND playing_date = ? LIMIT 1");
        $stmt->execute([$game['id'], $resultDate]);
        
        return $stmt->fetch() ? "Open" : "Result Pending";
        
    } else {
        $currentMinutes = timeToMinutes(date('H:i'));
        $closeMinutes = timeToMinutes($game['close_time']);
        
        if ($currentMinutes >= $closeMinutes) {
            return "Closed";
        }
        
        if (isFirstDayOfMonth()) {
            return "Open";
        }
        
        $resultDate = getNormalGameResultDate();
        $stmt = $pdo->prepare("SELECT id FROM results WHERE game_id = ? AND playing_date = ? LIMIT 1");
        $stmt->execute([$game['id'], $resultDate]);
        
        return $stmt->fetch() ? "Open" : "Result Pending";
    }
}
?>
