<?php
/**
 * Referral API - Get referral code, stats, referrals list
 * Endpoints:
 *   GET /api/referral.php?action=stats
 *   GET /api/referral.php?action=list
 *   GET /api/referral.php?action=validate&code=XXXXXX
 */

require_once 'config.php';

$action = $_GET['action'] ?? 'stats';

switch ($action) {
    case 'stats':
        handleReferralStats($pdo);
        break;
    case 'list':
        handleReferralList($pdo);
        break;
    case 'validate':
        handleValidateCode($pdo);
        break;
    case 'share_text':
        handleShareText($pdo);
        break;
    default:
        sendError('Invalid action', 400);
}

/**
 * Get Referral Stats
 */
function handleReferralStats($pdo) {
    $user = authenticate($pdo);
    
    // Get referral code (generate if not exists)
    $referralCode = $user['referral_code'];
    if (empty($referralCode)) {
        $referralCode = strtoupper(substr(md5($user['mobile'] . time()), 0, 6));
        $stmt = $pdo->prepare("UPDATE users SET referral_code = ? WHERE id = ?");
        $stmt->execute([$referralCode, $user['id']]);
    }
    
    // Get total referrals
    $stmt = $pdo->prepare("SELECT COUNT(*) as total FROM referrals WHERE referrer_id = ?");
    $stmt->execute([$user['id']]);
    $totalReferrals = $stmt->fetchColumn();
    
    // Get successful referrals (bonus given)
    $stmt = $pdo->prepare("SELECT COUNT(*) as successful FROM referrals WHERE referrer_id = ? AND bonus_given = 1");
    $stmt->execute([$user['id']]);
    $successfulReferrals = $stmt->fetchColumn();
    
    // Get total earned from referrals
    $stmt = $pdo->prepare("SELECT COALESCE(SUM(referrer_bonus), 0) as total_earned FROM referrals WHERE referrer_id = ? AND bonus_given = 1");
    $stmt->execute([$user['id']]);
    $totalEarned = $stmt->fetchColumn();
    
    // Get pending referrals (signed up but not deposited)
    $stmt = $pdo->prepare("SELECT COUNT(*) as pending FROM referrals WHERE referrer_id = ? AND bonus_given = 0");
    $stmt->execute([$user['id']]);
    $pendingReferrals = $stmt->fetchColumn();
    
    // Get bonus amounts from settings
    $referrerBonus = getSetting($pdo, 'referral_bonus_referrer', '50');
    $referredBonus = getSetting($pdo, 'referral_bonus_referred', '25');
    
    sendResponse([
        'referral_code' => $referralCode,
        'stats' => [
            'total_referrals' => (int)$totalReferrals,
            'successful_referrals' => (int)$successfulReferrals,
            'pending_referrals' => (int)$pendingReferrals,
            'total_earned' => (float)$totalEarned
        ],
        'bonus_info' => [
            'you_get' => (int)$referrerBonus,
            'friend_gets' => (int)$referredBonus,
            'currency' => '₹'
        ]
    ]);
}

/**
 * Get List of Referrals
 */
function handleReferralList($pdo) {
    $user = authenticate($pdo);
    
    $limit = min(intval($_GET['limit'] ?? 50), 100);
    $page = max(intval($_GET['page'] ?? 1), 1);
    $offset = ($page - 1) * $limit;
    
    // Get total
    $stmt = $pdo->prepare("SELECT COUNT(*) FROM referrals WHERE referrer_id = ?");
    $stmt->execute([$user['id']]);
    $total = $stmt->fetchColumn();
    
    // Get referrals with user info
    $stmt = $pdo->prepare("
        SELECT r.*, u.name, u.mobile, u.created_at as joined_at, u.status as user_status
        FROM referrals r
        JOIN users u ON r.referred_id = u.id
        WHERE r.referrer_id = ?
        ORDER BY r.created_at DESC
        LIMIT ? OFFSET ?
    ");
    $stmt->execute([$user['id'], $limit, $offset]);
    $referrals = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    $referralList = [];
    foreach ($referrals as $ref) {
        $referralList[] = [
            'id' => (int)$ref['id'],
            'name' => $ref['name'],
            'mobile' => maskMobile($ref['mobile']),
            'joined_at' => $ref['joined_at'],
            'bonus_given' => (bool)$ref['bonus_given'],
            'bonus_amount' => (float)$ref['referrer_bonus'],
            'status' => $ref['bonus_given'] ? 'Completed' : 'Pending First Deposit'
        ];
    }
    
    sendResponse([
        'referrals' => $referralList,
        'pagination' => [
            'current_page' => $page,
            'per_page' => $limit,
            'total' => (int)$total,
            'total_pages' => ceil($total / $limit)
        ]
    ]);
}

/**
 * Validate Referral Code (for registration)
 */
function handleValidateCode($pdo) {
    $code = strtoupper(trim($_GET['code'] ?? ''));
    
    if (empty($code)) {
        sendError('Referral code required', 400);
    }
    
    $stmt = $pdo->prepare("SELECT id, name FROM users WHERE referral_code = ? AND status = 'active'");
    $stmt->execute([$code]);
    $referrer = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$referrer) {
        sendResponse([
            'valid' => false,
            'message' => 'Invalid referral code'
        ]);
    }
    
    $referredBonus = getSetting($pdo, 'referral_bonus_referred', '25');
    
    sendResponse([
        'valid' => true,
        'referrer_name' => $referrer['name'],
        'bonus' => (int)$referredBonus,
        'message' => "You'll get ₹{$referredBonus} bonus on your first deposit!"
    ]);
}

/**
 * Get Share Text for various platforms
 */
function handleShareText($pdo) {
    $user = authenticate($pdo);
    
    $referralCode = $user['referral_code'];
    $referredBonus = getSetting($pdo, 'referral_bonus_referred', '25');
    
    // Build referral link (adjust domain as needed)
    $protocol = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ? 'https' : 'http';
    $domain = $_SERVER['HTTP_HOST'] ?? 'satta-king.app';
    $referralLink = "{$protocol}://{$domain}/register.php?ref={$referralCode}";
    
    $whatsappText = "🎰 Join Satta King and get ₹{$referredBonus} bonus!\n\nUse my referral code: {$referralCode}\n\nDownload now: {$referralLink}";
    $telegramText = "🎰 Join Satta King and get ₹{$referredBonus} bonus! Use code: {$referralCode}";
    $smsText = "Join Satta King! Use code {$referralCode} and get ₹{$referredBonus} bonus. Link: {$referralLink}";
    
    sendResponse([
        'referral_code' => $referralCode,
        'referral_link' => $referralLink,
        'share_text' => [
            'whatsapp' => $whatsappText,
            'telegram' => $telegramText,
            'sms' => $smsText,
            'generic' => "Join Satta King using my code {$referralCode} and get ₹{$referredBonus} bonus! {$referralLink}"
        ]
    ]);
}
?>
