<?php
/**
 * Games API - List games, Game details, Results
 * Endpoints:
 *   GET /api/games.php?action=list
 *   GET /api/games.php?action=detail&id=1
 *   GET /api/games.php?action=results&id=1
 *   GET /api/games.php?action=today_results
 */

require_once 'config.php';

$action = $_GET['action'] ?? 'list';

switch ($action) {
    case 'list':
        handleGamesList($pdo);
        break;
    case 'detail':
        handleGameDetail($pdo);
        break;
    case 'results':
        handleGameResults($pdo);
        break;
    case 'today_results':
        handleTodayResults($pdo);
        break;
    case 'rates':
        handleGameRates($pdo);
        break;
    default:
        sendError('Invalid action', 400);
}

/**
 * Get All Active Games with Status
 */
function handleGamesList($pdo) {
    // Auth is optional for games list
    $user = null;
    $token = getBearerToken();
    if ($token) {
        $payload = verifyJWT($token);
        if ($payload) {
            $stmt = $pdo->prepare("SELECT * FROM users WHERE id = ? AND status = 'active'");
            $stmt->execute([$payload['user_id']]);
            $user = $stmt->fetch(PDO::FETCH_ASSOC);
        }
    }
    
    $stmt = $pdo->query("SELECT * FROM games WHERE status = 'active' ORDER BY sort_order ASC, name ASC");
    $games = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    $gamesList = [];
    foreach ($games as $game) {
        $isOpen = isGameOpenForApi($pdo, $game);
        $playingDate = getGamePlayingDateApi($game);
        $todayResult = getTodayResultApi($pdo, $game['id'], $playingDate);
        
        $gamesList[] = [
            'id' => (int)$game['id'],
            'name' => $game['name'],
            'open_time' => $game['open_time'],
            'close_time' => $game['close_time'],
            'is_overnight' => (bool)$game['is_overnight'],
            'is_open' => $isOpen,
            'playing_date' => $playingDate,
            'today_result' => $todayResult,
            'status_text' => $isOpen ? 'OPEN' : 'CLOSED',
            'status_color' => $isOpen ? '#00ff88' : '#ff4444'
        ];
    }
    
    $response = ['games' => $gamesList];
    
    // Add user balance if authenticated
    if ($user) {
        $response['balance'] = (float)getWalletBalance($pdo, $user['id']);
    }
    
    sendResponse($response);
}

/**
 * Get Single Game Detail
 */
function handleGameDetail($pdo) {
    $gameId = intval($_GET['id'] ?? 0);
    
    if (!$gameId) {
        sendError('Game ID required', 400);
    }
    
    $stmt = $pdo->prepare("SELECT * FROM games WHERE id = ? AND status = 'active'");
    $stmt->execute([$gameId]);
    $game = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$game) {
        sendError('Game not found', 404);
    }
    
    $isOpen = isGameOpenForApi($pdo, $game);
    $playingDate = getGamePlayingDateApi($game);
    $todayResult = getTodayResultApi($pdo, $game['id'], $playingDate);
    
    // Get recent results
    $stmt = $pdo->prepare("SELECT playing_date, result FROM results WHERE game_id = ? ORDER BY playing_date DESC LIMIT 7");
    $stmt->execute([$gameId]);
    $recentResults = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Get game rates
    $rate = getSetting($pdo, 'jodi_rate', '95');
    
    sendResponse([
        'game' => [
            'id' => (int)$game['id'],
            'name' => $game['name'],
            'open_time' => $game['open_time'],
            'close_time' => $game['close_time'],
            'is_overnight' => (bool)$game['is_overnight'],
            'is_open' => $isOpen,
            'playing_date' => $playingDate,
            'today_result' => $todayResult,
            'status_text' => $isOpen ? 'OPEN' : 'CLOSED'
        ],
        'rates' => [
            'jodi' => (int)$rate,
            'andar' => round((int)$rate / 10, 1),
            'bahar' => round((int)$rate / 10, 1)
        ],
        'recent_results' => $recentResults
    ]);
}

/**
 * Get Game Results History
 */
function handleGameResults($pdo) {
    $gameId = intval($_GET['id'] ?? 0);
    $limit = min(intval($_GET['limit'] ?? 30), 100);
    $page = max(intval($_GET['page'] ?? 1), 1);
    $offset = ($page - 1) * $limit;
    
    if (!$gameId) {
        sendError('Game ID required', 400);
    }
    
    // Get game name
    $stmt = $pdo->prepare("SELECT name FROM games WHERE id = ?");
    $stmt->execute([$gameId]);
    $game = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$game) {
        sendError('Game not found', 404);
    }
    
    // Get total count
    $stmt = $pdo->prepare("SELECT COUNT(*) FROM results WHERE game_id = ?");
    $stmt->execute([$gameId]);
    $total = $stmt->fetchColumn();
    
    // Get results
    $stmt = $pdo->prepare("SELECT playing_date, result, created_at FROM results WHERE game_id = ? ORDER BY playing_date DESC LIMIT ? OFFSET ?");
    $stmt->execute([$gameId, $limit, $offset]);
    $results = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    sendResponse([
        'game_name' => $game['name'],
        'results' => $results,
        'pagination' => [
            'current_page' => $page,
            'per_page' => $limit,
            'total' => (int)$total,
            'total_pages' => ceil($total / $limit)
        ]
    ]);
}

/**
 * Get Today's All Results
 */
function handleTodayResults($pdo) {
    $date = $_GET['date'] ?? date('Y-m-d');
    
    $stmt = $pdo->prepare("
        SELECT g.id, g.name, g.open_time, g.close_time, r.result, r.playing_date
        FROM games g
        LEFT JOIN results r ON g.id = r.game_id AND r.playing_date = ?
        WHERE g.status = 'active'
        ORDER BY g.sort_order ASC, g.name ASC
    ");
    $stmt->execute([$date]);
    $results = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    $gameResults = [];
    foreach ($results as $row) {
        $gameResults[] = [
            'game_id' => (int)$row['id'],
            'game_name' => $row['name'],
            'open_time' => $row['open_time'],
            'close_time' => $row['close_time'],
            'result' => $row['result'] ?? '--',
            'has_result' => !empty($row['result'])
        ];
    }
    
    sendResponse([
        'date' => $date,
        'results' => $gameResults
    ]);
}

/**
 * Get Game Rates/Settings
 */
function handleGameRates($pdo) {
    $jodiRate = getSetting($pdo, 'jodi_rate', '95');
    $minBid = getSetting($pdo, 'min_bid', '10');
    $maxBid = getSetting($pdo, 'max_bid', '10000');
    $dailyBidLimit = getSetting($pdo, 'daily_bid_limit', '100000');
    
    sendResponse([
        'rates' => [
            'jodi' => (int)$jodiRate,
            'andar' => round((int)$jodiRate / 10, 1),
            'bahar' => round((int)$jodiRate / 10, 1)
        ],
        'limits' => [
            'min_bid' => (int)$minBid,
            'max_bid' => (int)$maxBid,
            'daily_limit' => (int)$dailyBidLimit
        ]
    ]);
}

/**
 * Helper: Check if game is open (API version)
 */
function isGameOpenForApi($pdo, $game) {
    $now = new DateTime();
    $currentTime = $now->format('H:i:s');
    $currentDate = $now->format('Y-m-d');
    
    $openTime = $game['open_time'];
    $closeTime = $game['close_time'];
    $isOvernight = $game['is_overnight'];
    
    // Check if result already published for today's playing date
    $playingDate = getGamePlayingDateApi($game);
    $stmt = $pdo->prepare("SELECT id FROM results WHERE game_id = ? AND playing_date = ?");
    $stmt->execute([$game['id'], $playingDate]);
    if ($stmt->fetch()) {
        return false; // Result already published
    }
    
    if ($isOvernight) {
        // Overnight game: opens in morning, closes next day early morning
        if ($currentTime >= $openTime || $currentTime < $closeTime) {
            return true;
        }
    } else {
        // Regular game
        if ($currentTime >= $openTime && $currentTime < $closeTime) {
            return true;
        }
    }
    
    return false;
}

/**
 * Helper: Get playing date for game (API version)
 */
function getGamePlayingDateApi($game) {
    $now = new DateTime();
    $currentTime = $now->format('H:i:s');
    
    if ($game['is_overnight']) {
        // For overnight games, if current time is before close time (early morning),
        // the playing date is yesterday
        if ($currentTime < $game['close_time']) {
            $now->modify('-1 day');
        }
    }
    
    return $now->format('Y-m-d');
}

/**
 * Helper: Get today's result for game
 */
function getTodayResultApi($pdo, $gameId, $playingDate) {
    $stmt = $pdo->prepare("SELECT result FROM results WHERE game_id = ? AND playing_date = ?");
    $stmt->execute([$gameId, $playingDate]);
    $result = $stmt->fetch(PDO::FETCH_ASSOC);
    return $result ? $result['result'] : null;
}
?>
