<?php
/**
 * Auth API - Login, Register, Logout, Forgot Password
 * Endpoints:
 *   POST /api/auth.php?action=login
 *   POST /api/auth.php?action=register
 *   POST /api/auth.php?action=logout
 *   POST /api/auth.php?action=check
 */

require_once 'config.php';

$action = $_GET['action'] ?? $_POST['action'] ?? '';

switch ($action) {
    case 'login':
        handleLogin($pdo);
        break;
    case 'register':
        handleRegister($pdo);
        break;
    case 'logout':
        handleLogout($pdo);
        break;
    case 'check':
        handleCheckAuth($pdo);
        break;
    case 'refresh':
        handleRefreshToken($pdo);
        break;
    default:
        sendError('Invalid action', 400);
}

/**
 * Handle Login
 */
function handleLogin($pdo) {
    $data = getJsonInput();
    validateRequired($data, ['mobile', 'password']);
    
    $mobile = trim($data['mobile']);
    $password = $data['password'];
    
    // Rate limiting
    $ip = getClientIP();
    if (!checkApiRateLimit($pdo, 'login_' . $ip, 10, 300)) {
        sendError('Too many login attempts. Please try again later.', 429);
    }
    
    // Check if account is locked
    $stmt = $pdo->prepare("SELECT * FROM users WHERE mobile = ?");
    $stmt->execute([$mobile]);
    $user = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$user) {
        sendError('Invalid mobile number or password', 401);
    }
    
    // Check if account is locked
    if ($user['locked_until'] && strtotime($user['locked_until']) > time()) {
        $remaining = ceil((strtotime($user['locked_until']) - time()) / 60);
        sendError("Account is locked. Try again in {$remaining} minutes.", 403);
    }
    
    // Verify password
    if (!password_verify($password, $user['password'])) {
        // Increment failed attempts
        $failedAttempts = ($user['failed_login_attempts'] ?? 0) + 1;
        $maxAttempts = 5;
        
        if ($failedAttempts >= $maxAttempts) {
            $lockUntil = date('Y-m-d H:i:s', strtotime('+30 minutes'));
            $stmt = $pdo->prepare("UPDATE users SET failed_login_attempts = ?, locked_until = ? WHERE id = ?");
            $stmt->execute([$failedAttempts, $lockUntil, $user['id']]);
            sendError('Too many failed attempts. Account locked for 30 minutes.', 403);
        } else {
            $stmt = $pdo->prepare("UPDATE users SET failed_login_attempts = ? WHERE id = ?");
            $stmt->execute([$failedAttempts, $user['id']]);
            $remaining = $maxAttempts - $failedAttempts;
            sendError("Invalid password. {$remaining} attempts remaining.", 401);
        }
    }
    
    // Check user status
    if ($user['status'] === 'pending') {
        sendError('Your account is pending approval. Please wait.', 403);
    }
    
    if ($user['status'] === 'blocked') {
        sendError('Your account has been blocked. Contact support.', 403);
    }
    
    if ($user['status'] === 'rejected') {
        sendError('Your account registration was rejected.', 403);
    }
    
    if ($user['is_admin'] == 1) {
        sendError('Admin accounts cannot login via app', 403);
    }
    
    // Reset failed attempts and update last login
    $stmt = $pdo->prepare("UPDATE users SET failed_login_attempts = 0, locked_until = NULL, last_login = NOW(), last_login_ip = ? WHERE id = ?");
    $stmt->execute([$ip, $user['id']]);
    
    // Generate JWT token
    $token = generateJWT($user['id'], $user['mobile']);
    
    // Get wallet balance
    $balance = getWalletBalance($pdo, $user['id']);
    
    sendResponse([
        'token' => $token,
        'user' => [
            'id' => (int)$user['id'],
            'name' => $user['name'],
            'mobile' => $user['mobile'],
            'status' => $user['status'],
            'balance' => (float)$balance,
            'referral_code' => $user['referral_code'],
            'has_pin' => !empty($user['transaction_pin']),
            'profile_photo' => $user['profile_photo'] ? '/uploads/' . $user['profile_photo'] : null,
            'created_at' => $user['created_at']
        ],
        'message' => 'Login successful'
    ]);
}

/**
 * Handle Register
 */
function handleRegister($pdo) {
    $data = getJsonInput();
    validateRequired($data, ['name', 'mobile', 'password']);
    
    $name = trim($data['name']);
    $mobile = trim($data['mobile']);
    $password = $data['password'];
    $referralCode = trim($data['referral_code'] ?? '');
    
    // Validate mobile
    if (!preg_match('/^[0-9]{10}$/', $mobile)) {
        sendError('Invalid mobile number. Must be 10 digits.', 400);
    }
    
    // Validate password
    if (strlen($password) < 6) {
        sendError('Password must be at least 6 characters.', 400);
    }
    
    // Check if mobile exists
    $stmt = $pdo->prepare("SELECT id FROM users WHERE mobile = ?");
    $stmt->execute([$mobile]);
    if ($stmt->fetch()) {
        sendError('Mobile number already registered.', 409);
    }
    
    // Validate referral code if provided
    $referredBy = null;
    if (!empty($referralCode)) {
        $stmt = $pdo->prepare("SELECT id FROM users WHERE referral_code = ? AND status = 'active'");
        $stmt->execute([$referralCode]);
        $referrer = $stmt->fetch();
        if (!$referrer) {
            sendError('Invalid referral code.', 400);
        }
        $referredBy = $referrer['id'];
    }
    
    // Generate unique referral code for new user
    $newReferralCode = strtoupper(substr(md5($mobile . time()), 0, 6));
    
    // Hash password
    $hashedPassword = password_hash($password, PASSWORD_DEFAULT);
    
    // Get default status from settings
    $requireApproval = getSetting($pdo, 'require_user_approval', '1');
    $status = $requireApproval == '1' ? 'pending' : 'active';
    
    // Insert user
    $stmt = $pdo->prepare("INSERT INTO users (name, mobile, password, status, referral_code, referred_by, created_at) VALUES (?, ?, ?, ?, ?, ?, NOW())");
    $stmt->execute([$name, $mobile, $hashedPassword, $status, $newReferralCode, $referredBy]);
    $userId = $pdo->lastInsertId();
    
    // Create referral record if referred
    if ($referredBy) {
        $stmt = $pdo->prepare("INSERT INTO referrals (referrer_id, referred_id, referral_code, created_at) VALUES (?, ?, ?, NOW())");
        $stmt->execute([$referredBy, $userId, $referralCode]);
    }
    
    // Add welcome notification
    $stmt = $pdo->prepare("INSERT INTO notifications (user_id, type, title, message, created_at) VALUES (?, 'welcome', 'Welcome to Satta King! 🎰', 'Your account has been created successfully. Good luck!', NOW())");
    $stmt->execute([$userId]);
    
    $response = [
        'user_id' => (int)$userId,
        'status' => $status,
        'referral_code' => $newReferralCode,
        'message' => $status === 'pending' 
            ? 'Registration successful! Your account is pending admin approval.' 
            : 'Registration successful! You can now login.'
    ];
    
    // If auto-approved, also send token
    if ($status === 'active') {
        $response['token'] = generateJWT($userId, $mobile);
        $response['user'] = [
            'id' => (int)$userId,
            'name' => $name,
            'mobile' => $mobile,
            'status' => $status,
            'balance' => 0,
            'referral_code' => $newReferralCode,
            'has_pin' => false
        ];
    }
    
    sendResponse($response, 201);
}

/**
 * Handle Logout
 */
function handleLogout($pdo) {
    $user = authenticate($pdo);
    
    // Optionally: You can implement token blacklisting here
    // For now, just return success (client should delete token)
    
    sendResponse([
        'message' => 'Logged out successfully'
    ]);
}

/**
 * Check Auth Token Validity
 */
function handleCheckAuth($pdo) {
    $user = authenticate($pdo);
    $balance = getWalletBalance($pdo, $user['id']);
    
    // Get unread notifications count
    $stmt = $pdo->prepare("SELECT COUNT(*) FROM notifications WHERE user_id = ? AND is_read = 0");
    $stmt->execute([$user['id']]);
    $unreadNotifications = $stmt->fetchColumn();
    
    sendResponse([
        'valid' => true,
        'user' => [
            'id' => (int)$user['id'],
            'name' => $user['name'],
            'mobile' => $user['mobile'],
            'status' => $user['status'],
            'balance' => (float)$balance,
            'referral_code' => $user['referral_code'],
            'has_pin' => !empty($user['transaction_pin']),
            'profile_photo' => $user['profile_photo'] ? '/uploads/' . $user['profile_photo'] : null,
            'unread_notifications' => (int)$unreadNotifications
        ]
    ]);
}

/**
 * Refresh Token
 */
function handleRefreshToken($pdo) {
    $user = authenticate($pdo);
    
    // Generate new token
    $token = generateJWT($user['id'], $user['mobile']);
    
    sendResponse([
        'token' => $token,
        'message' => 'Token refreshed successfully'
    ]);
}
?>
