<?php
require_once '../debug.php';
require_once '../config/db.php';
require_once '../config/auth.php';
require_once '../config/functions.php';
require_once '../config/security.php';

initSecureSession();
requireAdmin();

// Check session timeout
if (!checkSessionTimeout($pdo)) {
    header("Location: ../login.php?msg=Session+expired");
    exit();
}

$admin = getUser($pdo);

// Track admin session
trackUserSession($pdo, $admin['id']);

$error = '';
$success = '';

// Get all settings
// Get all settings
function getAllSettings($pdo) {
    $stmt = $pdo->query("SELECT * FROM settings");
    $settings = [];
    while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
        // Try different column names
        $key = $row['setting_key'] ?? $row['key'] ?? $row['name'] ?? null;
        $value = $row['setting_value'] ?? $row['value'] ?? null;
        if ($key !== null) {
            $settings[$key] = $value;
        }
    }
    return $settings;
}

// Save setting
function saveSetting($pdo, $key, $value) {
    // First check which column names exist
    $stmt = $pdo->query("DESCRIBE settings");
    $columns = $stmt->fetchAll(PDO::FETCH_COLUMN);
    
    // Determine column names
    if (in_array('setting_key', $columns)) {
        $keyCol = 'setting_key';
        $valCol = 'setting_value';
    } elseif (in_array('key', $columns)) {
        $keyCol = '`key`';
        $valCol = '`value`';
    } else {
        $keyCol = 'name';
        $valCol = 'value';
    }
    
    $stmt = $pdo->prepare("SELECT * FROM settings WHERE $keyCol = ?");
    $stmt->execute([$key]);
    
    if ($stmt->fetch()) {
        $stmt = $pdo->prepare("UPDATE settings SET $valCol = ? WHERE $keyCol = ?");
        $stmt->execute([$value, $key]);
    } else {
        $stmt = $pdo->prepare("INSERT INTO settings ($keyCol, $valCol) VALUES (?, ?)");
        $stmt->execute([$key, $value]);
    }
}

// Handle form submissions
if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    
    // Payment Settings
    if (isset($_POST['save_payment'])) {
        if (!verifyAdminCaptcha($_POST['captcha_answer'] ?? '')) {
            $error = getCaptchaError();
        } else {
            saveSetting($pdo, 'admin_upi', trim($_POST['admin_upi']));
            saveSetting($pdo, 'min_deposit', floatval($_POST['min_deposit']));
            saveSetting($pdo, 'max_deposit', floatval($_POST['max_deposit']));
            saveSetting($pdo, 'min_withdraw', floatval($_POST['min_withdraw']));
            saveSetting($pdo, 'max_withdraw', floatval($_POST['max_withdraw']));
            
            // Handle QR code upload
            if (!empty($_FILES['admin_qr']['name'])) {
                $allowed = ['jpg', 'jpeg', 'png', 'gif'];
                $ext = strtolower(pathinfo($_FILES['admin_qr']['name'], PATHINFO_EXTENSION));
                if (in_array($ext, $allowed)) {
                    $filename = 'admin_qr_' . time() . '.' . $ext;
                    $uploadPath = '../uploads/' . $filename;
                    if (move_uploaded_file($_FILES['admin_qr']['tmp_name'], $uploadPath)) {
                        saveSetting($pdo, 'admin_qr', $filename);
                    }
                }
            }
            
            $success = 'Payment settings saved!';
        }
    }
    
    // Bid Settings
    if (isset($_POST['save_bid'])) {
        if (!verifyAdminCaptcha($_POST['captcha_answer'] ?? '')) {
            $error = getCaptchaError();
        } else {
            saveSetting($pdo, 'min_bid', floatval($_POST['min_bid']));
            saveSetting($pdo, 'max_bid', floatval($_POST['max_bid']));
            saveSetting($pdo, 'default_rate', floatval($_POST['default_rate']));
            $success = 'Bid settings saved!';
        }
    }
    
    // Contact Settings
    if (isset($_POST['save_contact'])) {
        if (!verifyAdminCaptcha($_POST['captcha_answer'] ?? '')) {
            $error = getCaptchaError();
        } else {
            saveSetting($pdo, 'whatsapp_number', trim($_POST['whatsapp_number']));
            saveSetting($pdo, 'support_email', trim($_POST['support_email']));
            saveSetting($pdo, 'telegram_link', trim($_POST['telegram_link']));
            $success = 'Contact settings saved!';
        }
    }
    
    // App Settings
    if (isset($_POST['save_app'])) {
        if (!verifyAdminCaptcha($_POST['captcha_answer'] ?? '')) {
            $error = getCaptchaError();
        } else {
            saveSetting($pdo, 'app_name', trim($_POST['app_name']));
            saveSetting($pdo, 'app_tagline', trim($_POST['app_tagline']));
            saveSetting($pdo, 'maintenance_mode', isset($_POST['maintenance_mode']) ? '1' : '0');
            saveSetting($pdo, 'registration_open', isset($_POST['registration_open']) ? '1' : '0');
            saveSetting($pdo, 'auto_approve', isset($_POST['auto_approve']) ? '1' : '0');
            $success = 'App settings saved!';
        }
    }
    
    // Notice Settings
    if (isset($_POST['save_notice'])) {
        if (!verifyAdminCaptcha($_POST['captcha_answer'] ?? '')) {
            $error = getCaptchaError();
        } else {
            saveSetting($pdo, 'marquee_text', trim($_POST['marquee_text']));
            saveSetting($pdo, 'popup_message', trim($_POST['popup_message']));
            saveSetting($pdo, 'show_popup', isset($_POST['show_popup']) ? '1' : '0');
            $success = 'Notice settings saved!';
        }
    }
    
    // Change Admin Password
    if (isset($_POST['change_password'])) {
        if (!verifyAdminCaptcha($_POST['captcha_answer'] ?? '')) {
            $error = getCaptchaError();
        } else {
            $current = $_POST['current_password'];
            $new = $_POST['new_password'];
            $confirm = $_POST['confirm_password'];
            
            if (!password_verify($current, $admin['password'])) {
                $error = 'Current password is incorrect!';
            } elseif (strlen($new) < 6) {
                $error = 'New password must be at least 6 characters!';
            } elseif ($new !== $confirm) {
                $error = 'New passwords do not match!';
            } else {
                $hash = password_hash($new, PASSWORD_DEFAULT);
                $stmt = $pdo->prepare("UPDATE users SET password = ? WHERE id = ?");
                $stmt->execute([$hash, $admin['id']]);
                $success = 'Password changed successfully!';
            }
        }
    }
}

$settings = getAllSettings($pdo);

// For sidebar
$stmt = $pdo->query("SELECT COUNT(*) as c FROM users WHERE is_admin = 0 AND status = 'pending'");
$pendingUsers = $stmt->fetch()['c'];
$stmt = $pdo->query("SELECT COUNT(*) as total FROM wallet_requests WHERE status = 'pending'");
$pendingRequests = $stmt->fetch()['total'];
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Settings - Admin Panel</title>
    <style>
        * { margin: 0; padding: 0; box-sizing: border-box; }
        body { font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif; background: #0f0f0f; color: #fff; min-height: 100vh; }
        
        .admin-layout { display: flex; min-height: 100vh; }
        
        .sidebar { width: 260px; background: #1a1a1a; border-right: 1px solid #333; position: fixed; top: 0; left: 0; bottom: 0; overflow-y: auto; z-index: 100; }
        .sidebar-header { padding: 20px; border-bottom: 1px solid #333; text-align: center; }
        .sidebar-logo { font-size: 22px; font-weight: 800; color: #ffd700; }
        .sidebar-subtitle { font-size: 11px; color: #888; margin-top: 5px; }
        .sidebar-nav { padding: 15px 0; }
        .nav-item { display: flex; align-items: center; gap: 12px; padding: 14px 20px; color: #888; text-decoration: none; transition: all 0.3s; border-left: 3px solid transparent; }
        .nav-item:hover { background: rgba(255,255,255,0.05); color: #fff; }
        .nav-item.active { background: rgba(139,0,0,0.3); color: #ffd700; border-left-color: #ffd700; }
        .nav-icon { font-size: 20px; width: 24px; text-align: center; }
        .nav-text { font-size: 14px; font-weight: 500; }
        .nav-badge { margin-left: auto; background: #ff4444; color: #fff; padding: 2px 8px; border-radius: 10px; font-size: 11px; font-weight: 700; }
        
        .main-content { flex: 1; margin-left: 260px; padding: 20px; }
        
        .top-bar { display: flex; justify-content: space-between; align-items: center; margin-bottom: 25px; padding-bottom: 20px; border-bottom: 1px solid #333; }
        .page-title { font-size: 24px; font-weight: 700; }
        .page-title span { color: #ffd700; }
        
        .alert { padding: 15px; border-radius: 10px; margin-bottom: 20px; font-size: 14px; }
        .alert.success { background: rgba(0,255,0,0.2); border: 1px solid #00cc00; color: #00ff00; }
        .alert.error { background: rgba(255,0,0,0.2); border: 1px solid #ff4444; color: #ff6666; }
        
        /* Settings Grid */
        .settings-grid { display: grid; grid-template-columns: repeat(auto-fit, minmax(400px, 1fr)); gap: 25px; }
        
        .settings-card { background: #1a1a1a; border-radius: 15px; border: 1px solid #333; overflow: hidden; }
        .settings-header { padding: 20px; border-bottom: 1px solid #333; display: flex; align-items: center; gap: 12px; }
        .settings-icon { width: 45px; height: 45px; border-radius: 12px; display: flex; align-items: center; justify-content: center; font-size: 22px; }
        .settings-icon.payment { background: rgba(0,200,83,0.2); }
        .settings-icon.bid { background: rgba(255,215,0,0.2); }
        .settings-icon.contact { background: rgba(52,152,219,0.2); }
        .settings-icon.app { background: rgba(156,39,176,0.2); }
        .settings-icon.notice { background: rgba(255,152,0,0.2); }
        .settings-icon.password { background: rgba(255,68,68,0.2); }
        .settings-title { font-size: 18px; font-weight: 700; }
        .settings-subtitle { font-size: 12px; color: #888; }
        
        .settings-body { padding: 20px; }
        
        .form-group { margin-bottom: 20px; }
        .form-group:last-child { margin-bottom: 0; }
        .form-label { display: block; font-size: 13px; color: #888; margin-bottom: 8px; }
        .form-input { width: 100%; padding: 14px; background: #252525; border: 2px solid #333; border-radius: 10px; color: #fff; font-size: 14px; outline: none; }
        .form-input:focus { border-color: #ffd700; }
        .form-textarea { width: 100%; padding: 14px; background: #252525; border: 2px solid #333; border-radius: 10px; color: #fff; font-size: 14px; outline: none; min-height: 100px; resize: vertical; }
        .form-textarea:focus { border-color: #ffd700; }
        
        .form-row { display: grid; grid-template-columns: 1fr 1fr; gap: 15px; }
        
        .form-check { display: flex; align-items: center; gap: 12px; padding: 15px; background: #252525; border-radius: 10px; cursor: pointer; margin-bottom: 10px; }
        .form-check input { width: 20px; height: 20px; accent-color: #ffd700; cursor: pointer; }
        .form-check-label { font-size: 14px; cursor: pointer; }
        .form-check-desc { font-size: 11px; color: #888; margin-top: 3px; }
        
        .qr-preview { width: 150px; height: 150px; background: #252525; border-radius: 10px; display: flex; align-items: center; justify-content: center; margin-bottom: 15px; overflow: hidden; }
        .qr-preview img { max-width: 100%; max-height: 100%; }
        .qr-preview .placeholder { color: #555; font-size: 40px; }
        
        .file-input { display: none; }
        .file-label { display: inline-block; padding: 12px 20px; background: #333; border-radius: 8px; cursor: pointer; font-size: 13px; transition: all 0.3s; }
        .file-label:hover { background: #444; }
        
        .btn-save { width: 100%; padding: 14px; background: linear-gradient(135deg, #00c853 0%, #00a844 100%); border: none; border-radius: 10px; color: #fff; font-size: 14px; font-weight: 700; cursor: pointer; margin-top: 10px; }
        .btn-save:hover { opacity: 0.9; }
        .btn-save.purple { background: linear-gradient(135deg, #9c27b0 0%, #7b1fa2 100%); }
        .btn-save.orange { background: linear-gradient(135deg, #ff9800 0%, #f57c00 100%); }
        .btn-save.red { background: linear-gradient(135deg, #ff4444 0%, #cc0000 100%); }
        
        .input-hint { font-size: 11px; color: #666; margin-top: 5px; }
        
        /* Mobile Nav */
        .mobile-nav { display: none; position: fixed; bottom: 0; left: 0; right: 0; background: #1a1a1a; padding: 10px 0; border-top: 1px solid #333; z-index: 100; }
        .mobile-nav-inner { display: flex; justify-content: space-around; }
        .mobile-nav-item { text-align: center; color: #666; text-decoration: none; padding: 5px 10px; }
        .mobile-nav-item.active { color: #ffd700; }
        .mobile-nav-icon { font-size: 20px; display: block; margin-bottom: 3px; }
        .mobile-nav-text { font-size: 10px; }
        
        @media (max-width: 900px) {
            .sidebar { display: none; }
            .main-content { margin-left: 0; padding-bottom: 80px; }
            .mobile-nav { display: block; }
            .settings-grid { grid-template-columns: 1fr; }
        }
        
        @media (max-width: 500px) {
            .form-row { grid-template-columns: 1fr; }
            .settings-grid { gap: 15px; }
        }
    </style>
</head>
<body>
    <div class="admin-layout">
        <aside class="sidebar">
            <div class="sidebar-header">
                <div class="sidebar-logo">🎰 SATTA KING</div>
                <div class="sidebar-subtitle">Admin Panel</div>
            </div>
            <nav class="sidebar-nav">
                <a href="index.php" class="nav-item"><span class="nav-icon">📊</span><span class="nav-text">Dashboard</span></a>
                <a href="users.php" class="nav-item"><span class="nav-icon">👥</span><span class="nav-text">Users</span><?php if ($pendingUsers > 0): ?><span class="nav-badge"><?= $pendingUsers ?></span><?php endif; ?></a>
                <a href="games.php" class="nav-item"><span class="nav-icon">🎮</span><span class="nav-text">Games</span></a>
                <a href="bids.php" class="nav-item"><span class="nav-icon">🎯</span><span class="nav-text">All Bids</span></a>
                <a href="results.php" class="nav-item"><span class="nav-icon">🏆</span><span class="nav-text">Results</span></a>
                <a href="wallet.php" class="nav-item"><span class="nav-icon">💰</span><span class="nav-text">Wallet</span><?php if ($pendingRequests > 0): ?><span class="nav-badge"><?= $pendingRequests ?></span><?php endif; ?></a>
                <a href="settings.php" class="nav-item active"><span class="nav-icon">⚙️</span><span class="nav-text">Settings</span></a>
                <a href="../logout.php" class="nav-item"><span class="nav-icon">🚪</span><span class="nav-text">Logout</span></a>
            </nav>
        </aside>

        <main class="main-content">
            <div class="top-bar">
                <h1 class="page-title">⚙️ <span>Settings</span></h1>
            </div>

            <?php if ($success): ?>
                <div class="alert success"><?= $success ?></div>
            <?php endif; ?>
            <?php if ($error): ?>
                <div class="alert error"><?= $error ?></div>
            <?php endif; ?>

            <div class="settings-grid">
                <!-- Payment Settings -->
                <div class="settings-card">
                    <div class="settings-header">
                        <div class="settings-icon payment">💳</div>
                        <div>
                            <div class="settings-title">Payment Settings</div>
                            <div class="settings-subtitle">UPI, QR Code & Limits</div>
                        </div>
                    </div>
                    <div class="settings-body">
                        <form method="POST" enctype="multipart/form-data">
                            <div class="form-group">
                                <label class="form-label">Admin UPI ID</label>
                                <input type="text" name="admin_upi" class="form-input" value="<?= htmlspecialchars($settings['admin_upi'] ?? '') ?>" placeholder="yourname@upi">
                            </div>
                            <div class="form-group">
                                <label class="form-label">Payment QR Code</label>
                                <div class="qr-preview">
                                    <?php if (!empty($settings['admin_qr'])): ?>
                                        <img src="../uploads/<?= $settings['admin_qr'] ?>" alt="QR Code">
                                    <?php else: ?>
                                        <span class="placeholder">📷</span>
                                    <?php endif; ?>
                                </div>
                                <input type="file" name="admin_qr" id="qrInput" class="file-input" accept="image/*">
                                <label for="qrInput" class="file-label">📤 Upload QR Code</label>
                            </div>
                            <div class="form-row">
                                <div class="form-group">
                                    <label class="form-label">Min Deposit (₹)</label>
                                    <input type="number" name="min_deposit" class="form-input" value="<?= $settings['min_deposit'] ?? 100 ?>">
                                </div>
                                <div class="form-group">
                                    <label class="form-label">Max Deposit (₹)</label>
                                    <input type="number" name="max_deposit" class="form-input" value="<?= $settings['max_deposit'] ?? 50000 ?>">
                                </div>
                            </div>
                            <div class="form-row">
                                <div class="form-group">
                                    <label class="form-label">Min Withdraw (₹)</label>
                                    <input type="number" name="min_withdraw" class="form-input" value="<?= $settings['min_withdraw'] ?? 100 ?>">
                                </div>
                                <div class="form-group">
                                    <label class="form-label">Max Withdraw (₹)</label>
                                    <input type="number" name="max_withdraw" class="form-input" value="<?= $settings['max_withdraw'] ?? 50000 ?>">
                                </div>
                            </div>
                            <div class="form-group" style="margin-top:15px;">
                                <label class="form-label">🔢 Captcha: <?= generateAdminCaptcha('payment') ?></label>
                                <input type="number" name="captcha_answer" class="form-input" placeholder="Enter answer" required>
                            </div>
                            <button type="submit" name="save_payment" class="btn-save">💾 Save Payment Settings</button>
                        </form>
                    </div>
                </div>

                <!-- Bid Settings -->
                <div class="settings-card">
                    <div class="settings-header">
                        <div class="settings-icon bid">🎯</div>
                        <div>
                            <div class="settings-title">Bid Settings</div>
                            <div class="settings-subtitle">Min/Max Bid & Rate</div>
                        </div>
                    </div>
                    <div class="settings-body">
                        <form method="POST">
                            <div class="form-row">
                                <div class="form-group">
                                    <label class="form-label">Min Bid Amount (₹)</label>
                                    <input type="number" name="min_bid" class="form-input" value="<?= $settings['min_bid'] ?? 10 ?>">
                                </div>
                                <div class="form-group">
                                    <label class="form-label">Max Bid Amount (₹)</label>
                                    <input type="number" name="max_bid" class="form-input" value="<?= $settings['max_bid'] ?? 10000 ?>">
                                </div>
                            </div>
                            <div class="form-group">
                                <label class="form-label">Default Rate (for new games)</label>
                                <input type="number" name="default_rate" class="form-input" value="<?= $settings['default_rate'] ?? 95 ?>" step="0.1">
                                <div class="input-hint">Example: 95 means ₹10 bid wins ₹950</div>
                            </div>
                            <div class="form-group" style="margin-top:15px;">
                                <label class="form-label">🔢 Captcha: <?= generateAdminCaptcha('bid') ?></label>
                                <input type="number" name="captcha_answer" class="form-input" placeholder="Enter answer" required>
                            </div>
                            <button type="submit" name="save_bid" class="btn-save">💾 Save Bid Settings</button>
                        </form>
                    </div>
                </div>

                <!-- Contact Settings -->
                <div class="settings-card">
                    <div class="settings-header">
                        <div class="settings-icon contact">📞</div>
                        <div>
                            <div class="settings-title">Contact Settings</div>
                            <div class="settings-subtitle">WhatsApp, Email & Telegram</div>
                        </div>
                    </div>
                    <div class="settings-body">
                        <form method="POST">
                            <div class="form-group">
                                <label class="form-label">WhatsApp Number</label>
                                <input type="text" name="whatsapp_number" class="form-input" value="<?= htmlspecialchars($settings['whatsapp_number'] ?? '') ?>" placeholder="919876543210">
                                <div class="input-hint">With country code, no + sign</div>
                            </div>
                            <div class="form-group">
                                <label class="form-label">Support Email</label>
                                <input type="email" name="support_email" class="form-input" value="<?= htmlspecialchars($settings['support_email'] ?? '') ?>" placeholder="support@example.com">
                            </div>
                            <div class="form-group">
                                <label class="form-label">Telegram Link</label>
                                <input type="url" name="telegram_link" class="form-input" value="<?= htmlspecialchars($settings['telegram_link'] ?? '') ?>" placeholder="https://t.me/yourchannel">
                            </div>
                            <div class="form-group" style="margin-top:15px;"><label class="form-label">🔢 Captcha: <?= generateAdminCaptcha('contact') ?></label><input type="number" name="captcha_answer" class="form-input" placeholder="Enter answer" required></div>
                            <button type="submit" name="save_contact" class="btn-save purple">💾 Save Contact Settings</button>
                        </form>
                    </div>
                </div>

                <!-- App Settings -->
                <div class="settings-card">
                    <div class="settings-header">
                        <div class="settings-icon app">📱</div>
                        <div>
                            <div class="settings-title">App Settings</div>
                            <div class="settings-subtitle">Name, Maintenance & Registration</div>
                        </div>
                    </div>
                    <div class="settings-body">
                        <form method="POST">
                            <div class="form-group">
                                <label class="form-label">App Name</label>
                                <input type="text" name="app_name" class="form-input" value="<?= htmlspecialchars($settings['app_name'] ?? 'SATTA KING') ?>">
                            </div>
                            <div class="form-group">
                                <label class="form-label">App Tagline</label>
                                <input type="text" name="app_tagline" class="form-input" value="<?= htmlspecialchars($settings['app_tagline'] ?? 'Play & Win Big!') ?>">
                            </div>
                            <div class="form-group">
                                <label class="form-check">
                                    <input type="checkbox" name="registration_open" <?= ($settings['registration_open'] ?? '1') == '1' ? 'checked' : '' ?>>
                                    <div>
                                        <span class="form-check-label">Registration Open</span>
                                        <div class="form-check-desc">Allow new user registrations</div>
                                    </div>
                                </label>
                                <label class="form-check">
                                    <input type="checkbox" name="auto_approve" <?= ($settings['auto_approve'] ?? '0') == '1' ? 'checked' : '' ?>>
                                    <div>
                                        <span class="form-check-label">Auto Approve Users</span>
                                        <div class="form-check-desc">Automatically approve new users</div>
                                    </div>
                                </label>
                                <label class="form-check">
                                    <input type="checkbox" name="maintenance_mode" <?= ($settings['maintenance_mode'] ?? '0') == '1' ? 'checked' : '' ?>>
                                    <div>
                                        <span class="form-check-label">Maintenance Mode</span>
                                        <div class="form-check-desc">Disable site for users</div>
                                    </div>
                                </label>
                            </div>
                            <div class="form-group" style="margin-top:15px;"><label class="form-label">🔢 Captcha: <?= generateAdminCaptcha('app') ?></label><input type="number" name="captcha_answer" class="form-input" placeholder="Enter answer" required></div>
                            <button type="submit" name="save_app" class="btn-save purple">💾 Save App Settings</button>
                        </form>
                    </div>
                </div>

                <!-- Notice Settings -->
                <div class="settings-card">
                    <div class="settings-header">
                        <div class="settings-icon notice">📢</div>
                        <div>
                            <div class="settings-title">Notice Settings</div>
                            <div class="settings-subtitle">Marquee & Popup Messages</div>
                        </div>
                    </div>
                    <div class="settings-body">
                        <form method="POST">
                            <div class="form-group">
                                <label class="form-label">Marquee Text (Scrolling)</label>
                                <textarea name="marquee_text" class="form-textarea" placeholder="Welcome to Satta King! Play responsibly..."><?= htmlspecialchars($settings['marquee_text'] ?? '') ?></textarea>
                            </div>
                            <div class="form-group">
                                <label class="form-label">Popup Message</label>
                                <textarea name="popup_message" class="form-textarea" placeholder="Important announcement for all users..."><?= htmlspecialchars($settings['popup_message'] ?? '') ?></textarea>
                            </div>
                            <div class="form-group">
                                <label class="form-check">
                                    <input type="checkbox" name="show_popup" <?= ($settings['show_popup'] ?? '0') == '1' ? 'checked' : '' ?>>
                                    <div>
                                        <span class="form-check-label">Show Popup</span>
                                        <div class="form-check-desc">Display popup to users on login</div>
                                    </div>
                                </label>
                            </div>
                            <div class="form-group" style="margin-top:15px;"><label class="form-label">🔢 Captcha: <?= generateAdminCaptcha('notice') ?></label><input type="number" name="captcha_answer" class="form-input" placeholder="Enter answer" required></div>
                            <button type="submit" name="save_notice" class="btn-save orange">💾 Save Notice Settings</button>
                        </form>
                    </div>
                </div>

                <!-- Change Password -->
                <div class="settings-card">
                    <div class="settings-header">
                        <div class="settings-icon password">🔐</div>
                        <div>
                            <div class="settings-title">Change Password</div>
                            <div class="settings-subtitle">Update admin password</div>
                        </div>
                    </div>
                    <div class="settings-body">
                        <form method="POST">
                            <div class="form-group">
                                <label class="form-label">Current Password</label>
                                <input type="password" name="current_password" class="form-input" required>
                            </div>
                            <div class="form-group">
                                <label class="form-label">New Password</label>
                                <input type="password" name="new_password" class="form-input" minlength="6" required>
                            </div>
                            <div class="form-group">
                                <label class="form-label">Confirm New Password</label>
                                <input type="password" name="confirm_password" class="form-input" minlength="6" required>
                            </div>
                            <div class="form-group" style="margin-top:15px;"><label class="form-label">🔢 Captcha: <?= generateAdminCaptcha('password') ?></label><input type="number" name="captcha_answer" class="form-input" placeholder="Enter answer" required></div>
                            <button type="submit" name="change_password" class="btn-save red">🔐 Change Password</button>
                        </form>
                    </div>
                </div>
            </div>
        </main>
    </div>

    <!-- Mobile Nav -->
    <nav class="mobile-nav">
        <div class="mobile-nav-inner">
            <a href="index.php" class="mobile-nav-item"><span class="mobile-nav-icon">📊</span><span class="mobile-nav-text">Home</span></a>
            <a href="users.php" class="mobile-nav-item"><span class="mobile-nav-icon">👥</span><span class="mobile-nav-text">Users</span></a>
            <a href="results.php" class="mobile-nav-item"><span class="mobile-nav-icon">🏆</span><span class="mobile-nav-text">Results</span></a>
            <a href="wallet.php" class="mobile-nav-item"><span class="mobile-nav-icon">💰</span><span class="mobile-nav-text">Wallet</span></a>
            <a href="settings.php" class="mobile-nav-item active"><span class="mobile-nav-icon">⚙️</span><span class="mobile-nav-text">Settings</span></a>
        </div>
    </nav>

    <script>
        // Preview QR code before upload
        document.getElementById('qrInput').addEventListener('change', function(e) {
            const file = e.target.files[0];
            if (file) {
                const reader = new FileReader();
                reader.onload = function(e) {
                    document.querySelector('.qr-preview').innerHTML = '<img src="' + e.target.result + '" alt="QR Preview">';
                }
                reader.readAsDataURL(file);
            }
        });
    </script>
</body>
</html>