<?php
require_once '../config/db.php';
require_once '../config/auth.php';
require_once '../config/functions.php';
require_once '../config/security.php';

initSecureSession();
requireAdmin();

if (!checkSessionTimeout($pdo)) {
    header("Location: ../login.php?msg=Session+expired");
    exit();
}

$admin = getUser($pdo);

// Check permission
if (!hasPermission($pdo, $admin['id'], 'reports') && !hasPermission($pdo, $admin['id'], 'all')) {
    header("Location: index.php?error=Permission+denied");
    exit();
}

// Date range
$startDate = $_GET['start'] ?? date('Y-m-01'); // First day of current month
$endDate = $_GET['end'] ?? date('Y-m-d');

// Overall Stats
$stmt = $pdo->query("SELECT COUNT(*) FROM users WHERE is_admin = 0");
$totalUsers = $stmt->fetchColumn();

$stmt = $pdo->query("SELECT COUNT(*) FROM users WHERE is_admin = 0 AND status = 'active'");
$activeUsers = $stmt->fetchColumn();

// Date Range Stats
$stmt = $pdo->prepare("SELECT 
    COUNT(*) as total_bids,
    COALESCE(SUM(amount), 0) as total_bid_amount,
    COALESCE(SUM(CASE WHEN status = 'won' THEN 1 ELSE 0 END), 0) as total_wins,
    COALESCE(SUM(win_amount), 0) as total_payout
    FROM bids WHERE DATE(created_at) BETWEEN ? AND ?");
$stmt->execute([$startDate, $endDate]);
$bidStats = $stmt->fetch(PDO::FETCH_ASSOC);

$stmt = $pdo->prepare("SELECT 
    COALESCE(SUM(CASE WHEN type = 'deposit' AND status = 'approved' THEN amount ELSE 0 END), 0) as total_deposits,
    COALESCE(SUM(CASE WHEN type = 'withdraw' AND status = 'approved' THEN amount ELSE 0 END), 0) as total_withdrawals,
    COUNT(CASE WHEN status = 'pending' THEN 1 END) as pending_requests
    FROM wallet_requests WHERE DATE(created_at) BETWEEN ? AND ?");
$stmt->execute([$startDate, $endDate]);
$walletStats = $stmt->fetch(PDO::FETCH_ASSOC);

$stmt = $pdo->prepare("SELECT COUNT(*) FROM users WHERE is_admin = 0 AND DATE(created_at) BETWEEN ? AND ?");
$stmt->execute([$startDate, $endDate]);
$newUsers = $stmt->fetchColumn();

// Calculate profit
$grossProfit = $bidStats['total_bid_amount'] - $bidStats['total_payout'];
$netProfit = $walletStats['total_deposits'] - $walletStats['total_withdrawals'];

// Daily Breakdown (last 7 days)
$stmt = $pdo->query("SELECT 
    DATE(created_at) as date,
    COUNT(*) as bids,
    COALESCE(SUM(amount), 0) as bid_amount,
    COALESCE(SUM(win_amount), 0) as payout
    FROM bids 
    WHERE created_at >= DATE_SUB(CURDATE(), INTERVAL 7 DAY)
    GROUP BY DATE(created_at)
    ORDER BY date DESC");
$dailyBids = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Top Winners
$stmt = $pdo->prepare("SELECT 
    user_name_archive as name,
    user_mobile_archive as mobile,
    COUNT(*) as wins,
    SUM(win_amount) as total_won
    FROM bids 
    WHERE status = 'won' AND DATE(created_at) BETWEEN ? AND ?
    GROUP BY user_id, user_name_archive, user_mobile_archive
    ORDER BY total_won DESC
    LIMIT 10");
$stmt->execute([$startDate, $endDate]);
$topWinners = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Top Games
$stmt = $pdo->prepare("SELECT 
    game_name_archive as name,
    COUNT(*) as total_bids,
    SUM(amount) as total_amount,
    SUM(CASE WHEN status = 'won' THEN win_amount ELSE 0 END) as total_payout
    FROM bids 
    WHERE DATE(created_at) BETWEEN ? AND ?
    GROUP BY game_id, game_name_archive
    ORDER BY total_amount DESC
    LIMIT 10");
$stmt->execute([$startDate, $endDate]);
$topGames = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Security Events (last 24 hours)
$stmt = $pdo->query("SELECT event_type, COUNT(*) as count, severity 
    FROM security_log 
    WHERE created_at >= DATE_SUB(NOW(), INTERVAL 24 HOUR)
    GROUP BY event_type, severity
    ORDER BY count DESC
    LIMIT 10");
$securityEvents = $stmt->fetchAll(PDO::FETCH_ASSOC);
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Reports - Admin</title>
    <style>
        * { margin: 0; padding: 0; box-sizing: border-box; }
        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
            background: #0a0a0a;
            color: #fff;
            min-height: 100vh;
        }
        .header {
            background: linear-gradient(135deg, #1a1a2e 0%, #16213e 100%);
            padding: 20px;
            display: flex;
            justify-content: space-between;
            align-items: center;
            border-bottom: 1px solid #333;
        }
        .header-title { font-size: 20px; font-weight: 600; }
        .back-link { color: #ffd700; text-decoration: none; }
        .container { padding: 20px; max-width: 1200px; margin: 0 auto; }
        
        /* Date Filter */
        .date-filter {
            background: #1a1a1a;
            padding: 15px;
            border-radius: 12px;
            margin-bottom: 20px;
            display: flex;
            gap: 15px;
            flex-wrap: wrap;
            align-items: center;
        }
        .date-filter input {
            padding: 10px;
            border-radius: 8px;
            border: 1px solid #333;
            background: #0f0f0f;
            color: #fff;
        }
        .date-filter button {
            padding: 10px 20px;
            background: linear-gradient(135deg, #ffd700, #ff8c00);
            border: none;
            border-radius: 8px;
            color: #000;
            font-weight: bold;
            cursor: pointer;
        }
        .quick-dates {
            display: flex;
            gap: 10px;
        }
        .quick-dates a {
            padding: 8px 12px;
            background: #333;
            border-radius: 6px;
            color: #fff;
            text-decoration: none;
            font-size: 12px;
        }
        .quick-dates a:hover { background: #444; }
        
        /* Stats Grid */
        .stats-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 15px;
            margin-bottom: 30px;
        }
        .stat-card {
            background: linear-gradient(135deg, #1a1a2e 0%, #0f0f1a 100%);
            padding: 20px;
            border-radius: 12px;
            border: 1px solid #333;
        }
        .stat-label { color: #888; font-size: 13px; margin-bottom: 8px; }
        .stat-value { font-size: 28px; font-weight: bold; color: #ffd700; }
        .stat-value.green { color: #00ff88; }
        .stat-value.red { color: #ff4444; }
        .stat-sub { font-size: 12px; color: #888; margin-top: 5px; }
        
        /* Tables */
        .section { margin-bottom: 30px; }
        .section-title { font-size: 18px; color: #ffd700; margin-bottom: 15px; }
        .data-table {
            width: 100%;
            border-collapse: collapse;
            background: #1a1a1a;
            border-radius: 12px;
            overflow: hidden;
        }
        .data-table th, .data-table td {
            padding: 12px 15px;
            text-align: left;
            border-bottom: 1px solid #333;
        }
        .data-table th { background: #0f0f0f; color: #888; font-weight: 500; }
        .data-table tr:hover { background: #222; }
        .amount { font-family: monospace; }
        .amount.green { color: #00ff88; }
        .amount.red { color: #ff4444; }
        
        /* Responsive */
        @media (max-width: 768px) {
            .stats-grid { grid-template-columns: repeat(2, 1fr); }
            .data-table { font-size: 13px; }
            .data-table th, .data-table td { padding: 8px 10px; }
        }
    </style>
</head>
<body>
    <div class="header">
        <a href="index.php" class="back-link">← Back</a>
        <span class="header-title">📊 Reports & Analytics</span>
        <span></span>
    </div>
    
    <div class="container">
        <!-- Date Filter -->
        <form class="date-filter" method="GET">
            <input type="date" name="start" value="<?= $startDate ?>">
            <span>to</span>
            <input type="date" name="end" value="<?= $endDate ?>">
            <button type="submit">Apply</button>
            <div class="quick-dates">
                <a href="?start=<?= date('Y-m-d') ?>&end=<?= date('Y-m-d') ?>">Today</a>
                <a href="?start=<?= date('Y-m-d', strtotime('-7 days')) ?>&end=<?= date('Y-m-d') ?>">Last 7 Days</a>
                <a href="?start=<?= date('Y-m-01') ?>&end=<?= date('Y-m-d') ?>">This Month</a>
                <a href="?start=<?= date('Y-01-01') ?>&end=<?= date('Y-m-d') ?>">This Year</a>
            </div>
        </form>
        
        <!-- Stats Grid -->
        <div class="stats-grid">
            <div class="stat-card">
                <div class="stat-label">Total Bids</div>
                <div class="stat-value"><?= number_format($bidStats['total_bids']) ?></div>
            </div>
            <div class="stat-card">
                <div class="stat-label">Bid Amount</div>
                <div class="stat-value">₹<?= number_format($bidStats['total_bid_amount'], 0) ?></div>
            </div>
            <div class="stat-card">
                <div class="stat-label">Total Payout</div>
                <div class="stat-value red">₹<?= number_format($bidStats['total_payout'], 0) ?></div>
            </div>
            <div class="stat-card">
                <div class="stat-label">Gross Profit</div>
                <div class="stat-value <?= $grossProfit >= 0 ? 'green' : 'red' ?>">₹<?= number_format($grossProfit, 0) ?></div>
            </div>
            <div class="stat-card">
                <div class="stat-label">Deposits</div>
                <div class="stat-value green">₹<?= number_format($walletStats['total_deposits'], 0) ?></div>
            </div>
            <div class="stat-card">
                <div class="stat-label">Withdrawals</div>
                <div class="stat-value red">₹<?= number_format($walletStats['total_withdrawals'], 0) ?></div>
            </div>
            <div class="stat-card">
                <div class="stat-label">Net Flow</div>
                <div class="stat-value <?= $netProfit >= 0 ? 'green' : 'red' ?>">₹<?= number_format($netProfit, 0) ?></div>
            </div>
            <div class="stat-card">
                <div class="stat-label">New Users</div>
                <div class="stat-value"><?= $newUsers ?></div>
                <div class="stat-sub">Total: <?= $totalUsers ?> | Active: <?= $activeUsers ?></div>
            </div>
        </div>
        
        <!-- Daily Breakdown -->
        <div class="section">
            <div class="section-title">📅 Last 7 Days</div>
            <table class="data-table">
                <thead>
                    <tr>
                        <th>Date</th>
                        <th>Bids</th>
                        <th>Amount</th>
                        <th>Payout</th>
                        <th>Profit</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($dailyBids as $day): 
                        $profit = $day['bid_amount'] - $day['payout'];
                    ?>
                    <tr>
                        <td><?= date('d M (D)', strtotime($day['date'])) ?></td>
                        <td><?= number_format($day['bids']) ?></td>
                        <td class="amount">₹<?= number_format($day['bid_amount'], 0) ?></td>
                        <td class="amount red">₹<?= number_format($day['payout'], 0) ?></td>
                        <td class="amount <?= $profit >= 0 ? 'green' : 'red' ?>">₹<?= number_format($profit, 0) ?></td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
        
        <!-- Top Winners -->
        <div class="section">
            <div class="section-title">🏆 Top Winners</div>
            <table class="data-table">
                <thead>
                    <tr>
                        <th>#</th>
                        <th>Name</th>
                        <th>Mobile</th>
                        <th>Wins</th>
                        <th>Total Won</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($topWinners as $i => $winner): ?>
                    <tr>
                        <td><?= $i + 1 ?></td>
                        <td><?= htmlspecialchars($winner['name']) ?></td>
                        <td><?= htmlspecialchars($winner['mobile']) ?></td>
                        <td><?= $winner['wins'] ?></td>
                        <td class="amount green">₹<?= number_format($winner['total_won'], 0) ?></td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
        
        <!-- Top Games -->
        <div class="section">
            <div class="section-title">🎮 Game Performance</div>
            <table class="data-table">
                <thead>
                    <tr>
                        <th>Game</th>
                        <th>Bids</th>
                        <th>Amount</th>
                        <th>Payout</th>
                        <th>Profit</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($topGames as $game): 
                        $profit = $game['total_amount'] - $game['total_payout'];
                    ?>
                    <tr>
                        <td><?= htmlspecialchars($game['name']) ?></td>
                        <td><?= number_format($game['total_bids']) ?></td>
                        <td class="amount">₹<?= number_format($game['total_amount'], 0) ?></td>
                        <td class="amount red">₹<?= number_format($game['total_payout'], 0) ?></td>
                        <td class="amount <?= $profit >= 0 ? 'green' : 'red' ?>">₹<?= number_format($profit, 0) ?></td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
        
        <!-- Security Events -->
        <?php if (!empty($securityEvents)): ?>
        <div class="section">
            <div class="section-title">🔒 Security Events (24h)</div>
            <table class="data-table">
                <thead>
                    <tr>
                        <th>Event Type</th>
                        <th>Count</th>
                        <th>Severity</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($securityEvents as $event): ?>
                    <tr>
                        <td><?= htmlspecialchars($event['event_type']) ?></td>
                        <td><?= $event['count'] ?></td>
                        <td style="color: <?= $event['severity'] == 'high' || $event['severity'] == 'critical' ? '#ff4444' : ($event['severity'] == 'medium' ? '#ffd700' : '#888') ?>">
                            <?= strtoupper($event['severity']) ?>
                        </td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
        <?php endif; ?>
    </div>
</body>
</html>
